from functools import wraps, partial, reduce

def fail_to_empty(f):
    """Assumes that the wrapped function takes self and parse_in"""

    @wraps(f)
    def wrapped(self,  parse_in):
        if len(parse_in) == 0:
            return []
        return f(self, parse_in)
    return wrapped

def no_effects(f):
    @wraps(f)
    def wrapped(self, parse_in):
        return list(map(lambda x: (x[0], x[1], []), f(self, parse_in)))
    return wrapped

def and_then(pa, pb, parse_in):
    """Passes `parse_in' through pa and runs all of the results through pb"""
    return [(parse_out[0] + pout[0], pout[1], parse_out[2] + pout[2]) for parse_out in pa.parse(parse_in) for pout in pb.parse(parse_out[1])]

def or_else(pa, pb, parse_in):
    pout = pa.parse(parse_in)
    if len(pout) == 0:
        return pb.parse(parse_in)
    return pout

def and_also(pa, pb, parse_in):
    pouta = pa.parse(parse_in)
    poutb = pb.parse(parse_in)
    return pouta + poutb

def the_least_greedy(pa, pb, parse_in):
    pouta = pa.parse(parse_in)
    poutb = pb.parse(parse_in)
    if len(pouta) > 0:
        maxa = max(list(map(lambda x: len(x[0]), pouta)))
    else:
        return poutb
    if len(poutb) > 0:
        maxb = max(list(map(lambda x: len(x[0]), poutb)))
    else:
        return pouta
    return pouta if maxa <= maxb else poutb


class base:
    """Base for a tparse parser object"""

    def __init__(self):
        pass

    @no_effects
    def parse(self, parse_in):
        return [('', parse_in)]

    def __rshift__(self, pother):
        class Combined(base):
            def parse(s, parse_in):
                return and_then(self, pother, parse_in)
        return Combined()

    def __lshift__(self, parse_in):
        pout = self.parse(parse_in)
        pout = list(map(lambda x: (x[0], x[1], list(map(lambda f: f(), x[2]))), pout))
        return pout

    def __or__(self, pother):
        class Or(base):
            def parse(s, parse_in):
                return or_else(self, pother, parse_in)
        return Or()

    def __and__(self, pother):
        class And(base):
            def parse(s, parse_in):
                return and_also(self, pother, parse_in)
        return And()

    def __xor__(self, state):
        class Stateful(base):
            def parse(s, parse_in):
                return list(map(lambda x: (x[0], x[1], x[2] + [partial(state, x[0])]), self.parse(parse_in)))
        return Stateful()

    def __gt__(self, state):
        class StateStateful(base):
            def parse(s, parse_in):
                pout = self.parse(parse_in)
                return list(map(lambda x: (x[0], x[1], [lambda: state(list(map(lambda f: f(), x[2])))]), pout))
        return StateStateful()

    def __truediv__(self, pother):
        class LeastGreedy(base):
            def parse(s, parse_in):
                return the_least_greedy(self, pother, parse_in)
        return LeastGreedy()

class digit(base):
    """tparse parser for a single digit.
Regexp: [0-9]"""

    @fail_to_empty
    @no_effects
    def parse(self, parse_in):
        if parse_in[0] in ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9']:
            return [(parse_in[0], parse_in[1:])]
        return []

class number(base):
    """tparse parser for a number.
Regexp: [0-9]+"""

    @no_effects
    def parse(self, parse_in):
        return at_least_one(digit()).parse(parse_in)

class char(base):
    """tparse parser for a given character"""

    def __init__(self, char):
        self.char = char

    @fail_to_empty
    @no_effects
    def parse(self, parse_in):
        if parse_in[0] == self.char:
            return [(self.char, parse_in[1:])]
        return []

    def __invert__(self):
        class not_char(base):
            @fail_to_empty
            @no_effects
            def parse(s, parse_in):
                if parse_in[0] != self.char:
                    return [(parse_in[0], parse_in[1:])]
                return []
        return not_char()

class string(base):
    """tparse parser for a given string"""

    def __init__(self, string):
        self.string = string

    @fail_to_empty
    @no_effects
    def parse(self, parse_in):
        combined = reduce(lambda a, x: a >> x, map(char, self.string))
        return combined.parse(parse_in)

    def __invert__(self):
        class not_string(base):
            @fail_to_empty
            @no_effects
            def parse(s, parse_in):
                if self.string != parse_in[:len(self.string)]:
                    rest = s.parse(parse_in[1:])
                    rest_f = parse_in[0] + rest[0][0] if len(rest) > 0 else parse_in[0]
                    rest_s = rest[0][1] if len(rest) > 0 else parse_in[1:]
                    return [(rest_f, rest_s)]
                return []
        return not_string()

class space(base):
    """tparse parser for a blank space"""

    @no_effects
    def parse(self, parse_in):
        return char(' ').parse(parse_in)

class tab(base):
    """tparse parser for a tab character"""

    @no_effects
    def parse(self, parse_in):
        return char('\t').parse(parse_in)

class newline(base):
    """tparse parser for a newline character"""

    @no_effects
    def parse(self, parse_in):
        return char('\n').parse(parse_in)

class whitespace(base):
    """tparse parser for any whitespace character"""

    @no_effects
    def parse(self, parse_in):
        return (space() | tab() | newline()).parse(parse_in)

class any(base):
    """higher order parser that repeats a parser for any number of times"""

    def __init__(self, parser):
        self.parser = parser

    def parse(self, parse_in):
        return ((self.parser >> self) | base()).parse(parse_in)

class at_least_one(base):
    """higher order parser that repeats a parser for at least one time"""

    def __init__(self, parser):
        self.parser = parser

    @fail_to_empty
    def parse(self, parse_in):
        return ((self.parser >> self) | self.parser).parse(parse_in)

class maybe(base):
    """higher order parser that applies a parser zero or one time"""

    def __init__(self, parser):
        self.parser = parser

    def parse(self, parse_in):
        return (self.parser | base()).parse(parse_in)
