#!/usr/bin/env python
# coding=utf-8
from ConfigParser import ConfigParser
from .rpc_client import rpc_client
from . import __version__
import feedparser
import urlparse
import argparse
import hashlib
import urllib
import redis
import json
import time
import sys
import os

CONF_FILE = '~/.trmaid.conf'
CONF_FILE_FULL = os.path.expanduser(CONF_FILE)


def msg_and_exit(*msg):
    print '\n'.join(msg)
    sys.exit(1)


def utfme(text):
    if isinstance(text, unicode):
        text = text.encode('utf-8')
    return text


def url_hostname(url):
    return urlparse.urlparse(url).hostname


def normalize_dir(directory):
    while '//' in directory:
        directory = directory.replace('//', '/')
    return directory.rstrip('/')


class TRMaid(object):

    def __init__(self, config_file=CONF_FILE_FULL, verbose=False):
        if not os.path.exists(config_file):
            msg_and_exit(
                'Please create config file called %s.' % config_file,
                'Go see the example at:',
                '  https://github.com/tanelpuhu/trmaid',
            )

        # config
        self.conf = ConfigParser()
        self.conf.read(config_file)
        self.sections = self.conf.sections()

        # verbose
        self.verbose = verbose
        if self.conf.has_option('host', 'verbose'):
            self.verbose = (
                self.verbose or self.conf.getboolean('host', 'verbose')
            )

        self.msg('using config from %s' % config_file)

        # redis
        if (self.conf.has_section('host') and
            self.conf.has_option('host', 'redis')
            ):
            rds_url = self.conf.get('host', 'redis')
        else:
            rds_url = 'redis://localhost:6379'

        try:
            self.rds = redis.from_url(rds_url)
            self.rds.ping()
        except redis.exceptions.ConnectionError:
            msg_and_exit("Please install and run redis on %s" % rds_url)

        # transmission conf check
        if not self.conf.has_option('host', 'url'):
            msg_and_exit('host.url not configured')
        elif not self.conf.has_option('host', 'username'):
            msg_and_exit('host.username not configured')
        elif not self.conf.has_option('host', 'password'):
            msg_and_exit('host.password not configured')

        # rpc stuff
        verify = True
        if self.conf.has_option('host', 'verify'):
            verify = self.conf.getboolean('host', 'verify')

        self.rpc = rpc_client(
            self.conf.get('host', 'url'),
            self.conf.get('host', 'username'),
            self.conf.get('host', 'password'),
            verify=verify
        )

    def msg(self, *msgs):
        if self.verbose:
            print ' => ' + '\n    '.join(msgs)

    def verbose_func(self, func, *args):
        if self.verbose:
            print '%s(%s)' % (func.__name__, json.dumps(args, indent=2)),
        result = func(*args)
        if self.verbose:
            print '=>', json.dumps(result, indent=2)
        return result

    def conf_get(self, section, item, cast=None):
        if self.conf.has_option(section, item):
            if cast == bool:
                func = self.conf.getboolean
            elif cast == int:
                func = self.conf.getint
            elif cast == float:
                func = self.conf.getfloat
            else:
                func = self.conf.get
            return func(section, item)

    def handle_session(self):
        if not self.conf.has_section('session'):
            return self.msg('No [session] section')

        req = {}
        args = self.rpc.session_get().get('arguments', {})
        free = args.get('download-dir-free-space', 0)
        if free:
            units = float(args.get('units', {}).get('size-bytes', 1000))
            free_gb = free / (units * units * units)
            if free_gb < 50:
                self.msg('only %.2f Gb free space left!' % free_gb)

        options = [
            # Booleans
            (bool, [
                "alt-speed-enabled",
                "alt-speed-time-enabled",
                "blocklist-enabled",
                "dht-enabled",
                "download-queue-enabled",
                "idle-seeding-limit-enabled",
                "incomplete-dir-enabled",
                "lpd-enabled",
                "peer-port-random-on-start",
                "pex-enabled",
                "port-forwarding-enabled",
                "queue-stalled-enabled",
                "rename-partial-files",
                "script-torrent-done-enabled",
                "seed-queue-enabled",
                "seedRatioLimited",
                "speed-limit-down-enabled",
                "speed-limit-up-enabled",
                "start-added-torrents",
                "trash-original-torrent-files",
                "utp-enabled",
            ]),
            # Doubles / Floats
            (float, [
                "seedRatioLimit",
            ]),
            # Integers
            (int, [
             "alt-speed-down",
             "alt-speed-time-begin",
             "alt-speed-time-day",
             "alt-speed-time-end",
             "alt-speed-up",
             "cache-size-mb",
             "download-queue-size",
             "idle-seeding-limit",
             "peer-limit-global",
             "peer-limit-per-torrent",
             "peer-port",
             "queue-stalled-minutes",
             "seed-queue-size",
             "speed-limit-down",
             "speed-limit-up",
             ]),
            # Text / Strings
            (None, [
                "blocklist-url",
                "download-dir",
                "encryption",
                "incomplete-dir",
                "script-torrent-done-filename",
            ])
        ]
        for cast, keys in options:
            for key in keys:
                value = self.conf_get('session', key, cast=cast)
                if value is None:
                    continue
                current = args.get(key)
                if current != value:
                    req[key] = value

        # blocklist
        blocklist_enabled = self.conf_get(
            'session', 'blocklist-enabled', cast=bool
        )
        blocklist_update_interval = self.conf_get(
            'session', 'blocklist-update-interval', cast=int
        )
        blocklist_url = self.conf_get('session', 'blocklist-url')

        if blocklist_update_interval and blocklist_enabled and blocklist_url:
            if self.rds.setnx('trmaid:blocklist-update', '1'):
                self.rds.expire(
                    'trmaid:blocklist-update', blocklist_update_interval * 3600
                )
                self.verbose_func(self.rpc.blocklist_update)

        if req:
            self.verbose_func(self.rpc.session_set, req)

    def handle_all_torrents(self):
        torrents = self.rpc.torrent_get({'fields': [
            'bandwidthPriority',
            'doneDate',
            'downloadDir',
            'downloadLimit',
            'downloadLimited',
            'hashString',
            'honorsSessionLimits',
            'id',
            'isPrivate',
            'name',
            'peer-limit',
            'peer-limit-global',
            'peer-limit-per-torrent',
            'percentDone',
            'queuePosition',
            'seedIdleLimit',
            'seedIdleMode',
            'seedRatioLimit',
            'seedRatioMode',
            'status',
            'totalSize',
            'trackers',
            'uploadLimit',
            'uploadLimited',
            'uploadRatio',
        ]})
        if torrents.get('result') == 'success':
            torrents = torrents.get('arguments', {}).get('torrents', [])
            self.msg('there are %d total torrents...' % len(torrents))
            for tor in torrents:
                tor['status_text'] = self.rpc.get_status(tor['status'])
                tor['percentDone'] *= 100.0
                tor['name'] = utfme(tor['name'])
                self.handle_torrent(tor)

    def handle_torrent(self, tor):
        for tracker in tor['trackers']:
            announce = tracker.get('announce')
            if not announce:
                continue
            hostname = url_hostname(announce)
            if hostname not in self.sections:
                continue
            self.handle_torrent_section(tor, hostname)
            return

        location_section = 'location:%s' % normalize_dir(tor['downloadDir'])
        if self.conf.has_section(location_section):
            self.handle_torrent_section(tor, location_section)

        elif 'general-tracker' in self.sections:
            self.handle_torrent_section(tor, 'general-tracker')

    def handle_torrent_section(self, tor, section):
        for key in ['download-dir', 'location']:
            if self.conf.has_option(section, key):
                correct = normalize_dir(self.conf.get(section, key))
                current = normalize_dir(tor['downloadDir'])
                if current != correct:
                    self.verbose_func(
                        self.rpc.torrent_set_location, tor['id'], correct)
                    break

        if round(tor['percentDone'], 2) > 99.99:
            remove = []
            stopped = tor['status_text'] == 'Stopped'

            for hkey in ['remove-stopped-after-hours', 'remove-after-hours']:
                if self.conf.has_option(section, hkey):
                    if not stopped and hkey == 'remove-stopped-after-hours':
                        remove.append(0)
                        break
                    limit = self.conf.getfloat(section, hkey)
                    has_been = time.time() - tor['doneDate']
                    if has_been > limit * 3600:
                        remove.append(1)
                    else:
                        remove.append(0)
                        break

            for rkey in ['remove-stopped-after-ratio', 'remove-after-ratio']:
                if self.conf.has_option(section, rkey):
                    if not stopped and rkey == 'remove-stopped-after-ratio':
                        remove.append(0)
                        break
                    limit = self.conf.getfloat(section, rkey)
                    if tor['uploadRatio'] >= limit:
                        remove.append(1)
                    else:
                        remove.append(0)
                        break

            if remove and len(remove) == sum(remove):
                self.msg('removing %s...' % tor['name'])
                self.verbose_func(self.rpc.torrent_remove, tor['id'])
                return

        req = self.get_torrent_set_args(section, tor)
        if req:
            req['ids'] = tor['id']
            self.verbose_func(self.rpc.torrent_set, req)

    def get_torrent_add_args(self, section):
        req = {}
        options = [
            # Booleans
            (bool, [
                "paused",
            ]),
            # Doubles / Floats
            (float, [
                "seedRatioLimit",
            ]),
            # Integers
            (int, [
                "peer-limit",
                "bandwidthPriority",
            ]),
            # Text / Strings
            (None, [
                "cookies",
                "download-dir",
                "metainfo",
            ])
        ]
        for cast, keys in options:
            for key in keys:
                value = self.conf_get(section, key, cast=cast)
                if value is None:
                    continue
                req[key] = value

        return req

    def get_torrent_set_args(self, section, tor=None):
        req = {}
        options = [
            # Booleans
            (bool, [
                "downloadLimited",
                "honorsSessionLimits",
                "uploadLimited",
            ]),
            # Doubles / Floats
            (float, [
                'seedRatioLimit',
            ]),
            # Integers
            (int, [
                "bandwidthPriority",
                "downloadLimit",
                "peer-limit",
                "queuePosition",
                "seedIdleLimit",
                "seedIdleMode",
                "seedRatioMode",
                "uploadLimit",
            ]),
        ]
        for cast, keys in options:
            for key in keys:
                value = self.conf_get(section, key, cast=cast)
                if value is None:
                    continue
                if tor is None or tor[key] != value:
                    req[key] = value

        if self.conf.has_option(section, 'location'):
            location = self.conf.get(section, 'location')
            if tor is None or tor['downloadDir'] != location:
                req['location'] = location
        return req

    def add_links(self, section, links, rss=False):
        ids = []
        rdskey = 'trmaid:added:%s' % hashlib.md5(section).hexdigest()
        radded = rss and self.rds.lrange(rdskey, 0, -1) or []
        add_req = self.get_torrent_add_args(section)

        for link in links:
            xhash = hashlib.md5(link).hexdigest()
            if xhash in radded:
                continue
            add_req['filename'] = link
            res = self.verbose_func(self.rpc.torrent_add, add_req)
            result = res.get('result', '')

            if rss and result in ['success', 'duplicate torrent']:
                self.rds.lpush(rdskey, xhash)

            if result != 'success':
                continue

            arguments = res.get('arguments', {})

            if arguments.get('torrent-added', None):
                ids.append(arguments['torrent-added']['id'])

        if ids:
            if rss:
                self.rds.ltrim(rdskey, 0, 1000)
            set_req = self.get_torrent_set_args(section)
            set_req['ids'] = ids
            res = self.verbose_func(self.rpc.torrent_set, set_req)

    def handle_rss_and_redis_queues(self):
        for section in self.sections:
            if section.startswith('rss:'):
                self.handle_rss_section(section)
            elif section.startswith('redis:'):
                self.handle_redis_section(section)

    def handle_redis_section(self, section):
        queue = section[6:]
        self.msg('handling queue %s...' % queue[:40])

        def links():
            while 1:
                link = self.rds.lpop(queue)
                if not link:
                    break
                yield link
        return self.add_links(section, links())

    def handle_rss_section(self, section):
        feed_url = section[4:]
        self.msg('handling feed %s...' % feed_url[:40])

        def links():
            for entry in feedparser.parse(feed_url).entries:
                if not entry.link:
                    continue
                yield urllib.unquote(entry.link)

        return self.add_links(section, links(), rss=1)

    def run(self):
        self.handle_session()
        self.handle_all_torrents()
        self.handle_rss_and_redis_queues()


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-v',
        help='verbose',
        action='store_const',
        const=True,
        dest='verbose',
        default=False,
    )
    parser.add_argument(
        'config_file',
        help='specify config file other than %s' % CONF_FILE,
        nargs='?',
        default=CONF_FILE_FULL
    )
    parser.add_argument(
        '-V',
        help='Show version and exit',
        action='store_const',
        const=True,
        dest='version',
        default=False,
    )
    args = parser.parse_args()

    if args.version:
        print 'trmaid %s' % __version__
        return

    TRMaid(
        config_file=args.config_file,
        verbose=args.verbose
    ).run()

if __name__ == '__main__':
    main()
