#!/usr/bin/env python
# coding=utf-8
from .rpc_client import rpc_client
from . import __version__
from urlparse import urlparse
import feedparser
import argparse
import hashlib
import urllib
import redis
import types
import json
import yaml
import time
import sys
import os

CONF_FILE = '~/.trmaid.yaml'
CONF_FILE_FULL = os.path.expanduser(CONF_FILE)


def msg_and_exit(*msg):
    print '\n'.join(map(str, msg))
    sys.exit(1)


def utfme(elem):
    if isinstance(elem, types.DictType):
        for key in elem.keys():
            elem[key] = utfme(elem[key])
    elif isinstance(elem, (types.ListType, types.TupleType)):
        res = []
        for e in elem:
            res.append(utfme(e))
        elem = res
    elif isinstance(elem, basestring):
        elem = elem.encode('utf-8')
    return elem


def normalize_dir(directory):
    while '//' in directory:
        directory = directory.replace('//', '/')
    return directory.rstrip('/')


class TRMaid(object):

    def __init__(self, config_file=CONF_FILE_FULL, verbose=False):
        if not os.path.exists(config_file):
            msg_and_exit(
                'Please create config file called %s.' % config_file,
                'Go see the example at:',
                '  https://github.com/tanelpuhu/trmaid',
            )

        # config
        self.conf = yaml.load(open(CONF_FILE_FULL))
        general = self.conf.get('general', {})
        transmission = self.conf.get('transmission', {})

        # verbose
        self.verbose = verbose or general.get('verbose', False)

        self.msg('using config from %s' % config_file)

        # redis
        if 'redis' in general:
            rds_url = general['redis']
        else:
            rds_url = 'redis://localhost:6379'

        self.msg('redis url is %s' % rds_url)

        try:
            self.rds = redis.from_url(rds_url)
            self.rds.ping()
        except redis.exceptions.ConnectionError:
            msg_and_exit("Please install and run redis on %s" % rds_url)

        # transmission conf check
        if 'url' not in transmission:
            msg_and_exit('transmission.url not configured')
        elif 'username' not in transmission:
            msg_and_exit('transmission.username not configured')
        elif 'password' not in transmission:
            msg_and_exit('transmission.password not configured')

        # rpc stuff
        verify = general.get('verify', True)

        self.rpc = rpc_client(
            transmission['url'],
            transmission['username'],
            transmission['password'],
            verify=verify
        )

    def msg(self, *msgs):
        if self.verbose:
            print ' => ' + '\n    '.join(msgs)

    def verbose_func(self, func, *args):
        if self.verbose:
            print '%s(%s)' % (func.__name__, json.dumps(args, indent=2)),
        result = func(*args)
        if self.verbose:
            print '=>', json.dumps(result, indent=2)
        return result

    def conf_get(self, section, item, cast=None):
        if section in self.conf and item in self.conf[section]:
            return self.conf[section][item]

    def handle_session(self):
        if 'session' not in self.conf:
            return self.msg('No [session] section')

        req = {}
        args = self.rpc.session_get().get('arguments', {})

        free = args.get('download-dir-free-space', 0)
        if free:
            units = float(1024)
            free_gb = free / (units * units * units)
            free = args.get('download-dir-free-space', 0)
            if free_gb < 50:
                self.msg('ONLY %.2F GB FREE SPACE LEFT!!' % free_gb)
            else:
                self.msg('%.2f Gb free space left' % free_gb)

        options = [
            # Booleans
            (bool, [
                "alt-speed-enabled",
                "alt-speed-time-enabled",
                "blocklist-enabled",
                "dht-enabled",
                "download-queue-enabled",
                "idle-seeding-limit-enabled",
                "incomplete-dir-enabled",
                "lpd-enabled",
                "peer-port-random-on-start",
                "pex-enabled",
                "port-forwarding-enabled",
                "queue-stalled-enabled",
                "rename-partial-files",
                "script-torrent-done-enabled",
                "seed-queue-enabled",
                "seedRatioLimited",
                "speed-limit-down-enabled",
                "speed-limit-up-enabled",
                "start-added-torrents",
                "trash-original-torrent-files",
                "utp-enabled",
            ]),
            # Doubles / Floats
            (float, [
                "seedRatioLimit",
            ]),
            # Integers
            (int, [
             "alt-speed-down",
             "alt-speed-time-begin",
             "alt-speed-time-day",
             "alt-speed-time-end",
             "alt-speed-up",
             "cache-size-mb",
             "download-queue-size",
             "idle-seeding-limit",
             "peer-limit-global",
             "peer-limit-per-torrent",
             "peer-port",
             "queue-stalled-minutes",
             "seed-queue-size",
             "speed-limit-down",
             "speed-limit-up",
             ]),
            # Text / Strings
            (None, [
                "blocklist-url",
                "download-dir",
                "encryption",
                "incomplete-dir",
                "script-torrent-done-filename",
            ])
        ]
        session = self.conf['session']
        for cast, keys in options:
            for key in keys:
                value = session.get(key)
                if value is None:
                    continue
                current = args.get(key)
                if current != value:
                    req[key] = value

        # blocklist
        blocklist_enabled = session.get('blocklist-enabled')
        blocklist_update_interval = session.get('blocklist-update-interval')
        blocklist_url = session.get('blocklist-url')

        if blocklist_enabled and blocklist_update_interval and blocklist_url:
            if self.rds.setnx('trmaid:blocklist-update', '1'):
                self.rds.expire(
                    'trmaid:blocklist-update', blocklist_update_interval * 3600
                )
                # self.verbose_func(self.rpc.blocklist_update)

        if req:
            self.verbose_func(self.rpc.session_set, req)

    def handle_all_torrents(self):
        torrents = self.rpc.torrent_get({'fields': [
            'bandwidthPriority',
            'doneDate',
            'downloadDir',
            'downloadLimit',
            'downloadLimited',
            'hashString',
            'honorsSessionLimits',
            'id',
            'isPrivate',
            'name',
            'peer-limit',
            'peer-limit-global',
            'peer-limit-per-torrent',
            'percentDone',
            'queuePosition',
            'seedIdleLimit',
            'seedIdleMode',
            'seedRatioLimit',
            'seedRatioMode',
            'status',
            'totalSize',
            'trackers',
            'uploadLimit',
            'uploadLimited',
            'uploadRatio',
        ]})
        if torrents.get('result') == 'success':
            torrents = utfme(torrents.get('arguments', {}).get('torrents', []))
            self.msg('there are %d total torrents...' % len(torrents))
            for tor in torrents:
                tor['status_text'] = self.rpc.get_status(tor['status'])
                tor['percentDone'] *= 100.0
                self.handle_torrent(tor)

    def handle_torrent(self, tor):
        sections = self.conf.get('tracker', [])
        for tracker in tor['trackers']:
            announce = tracker.get('announce')
            if not announce:
                continue
            hostname = urlparse(announce).hostname
            if hostname not in sections:
                continue
            self.handle_torrent_section(tor, 'tracker', hostname)
            return

        sections = self.conf.get('location', [])
        location = normalize_dir(tor['downloadDir'])
        if location in sections:
            self.handle_torrent_section(tor, 'location', location)
            return

        general = self.conf.get('tracker', {}).get('general')
        if general:
            self.handle_torrent_section(tor, 'tracker', 'general')

    def handle_torrent_section(self, tor, section, subsection):
        conf = self.conf.get(section, {}).get(subsection, {})
        for key in ['download-dir', 'location']:
            if key in conf:
                correct = normalize_dir(conf.get(key))
                current = normalize_dir(tor['downloadDir'])
                if current != correct:
                    self.verbose_func(
                        self.rpc.torrent_set_location, tor['id'], correct)
                    break

        if round(tor['percentDone'], 2) > 99.99:
            remove = []
            stopped = tor['status_text'] == 'Stopped'
            for hkey in ['remove-stopped-after-hours', 'remove-after-hours']:
                if hkey in conf:
                    if not stopped and hkey == 'remove-stopped-after-hours':
                        remove.append(0)
                        break
                    limit = float(conf[hkey])
                    has_been = time.time() - tor['doneDate']
                    if has_been > limit * 3600:
                        remove.append(1)
                    else:
                        remove.append(0)
                        break

            for rkey in ['remove-stopped-after-ratio', 'remove-after-ratio']:
                if rkey in conf:
                    if not stopped and rkey == 'remove-stopped-after-ratio':
                        remove.append(0)
                        break
                    limit = float(conf[rkey])
                    if tor['uploadRatio'] >= limit:
                        remove.append(1)
                    else:
                        remove.append(0)
                        break

            if remove and len(remove) == sum(remove):
                self.msg('removing %s...' % tor['name'])
                self.verbose_func(self.rpc.torrent_remove, tor['id'])
                return

        req = self.get_torrent_set_args(section, subsection, tor)
        if req:
            req['ids'] = tor['id']
            self.verbose_func(self.rpc.torrent_set, req)

    def get_torrent_add_args(self, section, subsection):
        req = {}
        keys = [
            # Booleans
            "paused",
            # Doubles / Floats
            "seedRatioLimit",
            # Integers
            "peer-limit",
            "bandwidthPriority",
            # Text / Strings
            "cookies",
            "download-dir",
            "metainfo",
        ]
        for key in keys:
            value = self.conf.get(section, {}).get(subsection, {}).get(key)
            if value is None:
                continue
            req[key] = value

        return req

    def get_torrent_set_args(self, section, subsection, tor=None):
        req = {}
        keys = [
            # Booleans
            "downloadLimited",
            "honorsSessionLimits",
            "uploadLimited",
            # Doubles / Floats
            "seedRatioLimit",
            # Integers
            "bandwidthPriority",
            "downloadLimit",
            "peer-limit",
            "queuePosition",
            "seedIdleLimit",
            "seedIdleMode",
            "seedRatioMode",
            "uploadLimit",
        ]
        conf = self.conf.get(section, {}).get(subsection, {})
        for key in keys:
            value = conf.get(key)
            if value is None:
                continue
            if tor is None or tor[key] != value:
                req[key] = value

        if conf.get('location'):
            location = conf['location']
            if tor is None or tor['downloadDir'] != location:
                req['location'] = location

        if 'trackerAdd' in conf:
            req['trackerAdd'] = conf['trackerAdd']

        return req

    def add_links(self, section, subsection, links, rss=False):
        ids = []
        rdskey = 'trmaid:added:%s' % hashlib.md5(subsection).hexdigest()
        radded = rss and self.rds.lrange(rdskey, 0, -1) or []
        add_req = self.get_torrent_add_args(section, subsection)

        for link in links:
            xhash = hashlib.md5(link).hexdigest()
            if xhash in radded:
                continue
            add_req['filename'] = link
            res = self.verbose_func(self.rpc.torrent_add, add_req)
            result = res.get('result', '')

            if rss and result in ['success', 'duplicate torrent']:
                self.rds.lpush(rdskey, xhash)

            if result != 'success':
                continue

            arguments = res.get('arguments', {})

            if arguments.get('torrent-added', None):
                ids.append(arguments['torrent-added']['id'])

        if ids:
            if rss:
                self.rds.ltrim(rdskey, 0, 1000)
            set_req = self.get_torrent_set_args(section, subsection)
            set_req['ids'] = ids
            res = self.verbose_func(self.rpc.torrent_set, set_req)

    def handle_redis(self):
        def links(queue):
            while 1:
                link = self.rds.lpop(queue)
                if not link:
                    break
                yield link

        for queue in self.conf.get('redis', {}).keys():
            self.msg('handling queue %s...' % queue[:40])
            self.add_links('redis', queue, links(queue))

    def handle_rss(self):
        def links(feed_url):
            for entry in feedparser.parse(feed_url).entries:
                if not entry.link:
                    continue
                yield urllib.unquote(entry.link)

        for feed_url in self.conf.get('rss', {}).keys():
            self.msg('handling feed %s...' % feed_url[:40])
            self.add_links('rss', feed_url, links(feed_url), rss=True)

    def run(self):
        self.handle_session()
        self.handle_rss()
        self.handle_redis()
        self.handle_all_torrents()

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-v',
        help='verbose',
        action='store_const',
        const=True,
        dest='verbose',
        default=False,
    )
    parser.add_argument(
        'config_file',
        help='specify config file other than %s' % CONF_FILE,
        nargs='?',
        default=CONF_FILE_FULL
    )
    parser.add_argument(
        '-V',
        help='Show version and exit',
        action='store_const',
        const=True,
        dest='version',
        default=False,
    )
    args = parser.parse_args()

    if args.version:
        print 'trmaid %s' % __version__
        return

    TRMaid(
        config_file=args.config_file,
        verbose=args.verbose
    ).run()

if __name__ == '__main__':
    main()
