import requests
import json


class UnauthorizedException(Exception):
    "Unauthorized"


class NoSessionException(Exception):
    "NoSession"


class RPCClient(object):
    session_id = None
    headers = {}

    def __init__(self, url, username, password, verify=False):
        self.url = url
        self.username = username
        self.password = password
        # using self-signed certificate ;(
        self.verify = verify

    def make_request(self, data=None):
        if data is None:
            data = {}
        return requests.post(
            self.url,
            data=json.dumps(data),
            headers=self.get_req_headers(),
            verify=self.verify,
            auth=self.get_req_auth()
        ).json()

    def get_req_headers(self):
        return {
            'X-Transmission-Session-Id': self.get_session_id(),
        }

    def get_session_id(self):
        if not self.session_id:
            req = requests.post(
                self.url,
                verify=self.verify,
                auth=self.get_req_auth()
            )
            if req.status_code == 409:
                self.session_id = req.headers.get('x-transmission-session-id')
            elif req.status_code == 401:
                raise UnauthorizedException('Unauthorized User!')
            if not self.session_id:
                raise NoSessionException('No Session ID! Check your RPC url!')
        return self.session_id

    def get_req_auth(self):
        if self.username and self.password:
            return (self.username, self.password)

    @staticmethod
    def get_status(status):
        "turn numbers to text"
        status = str(status)
        return {
            '0': 'Stopped',
            '1': 'Queued to check',
            '2': 'Checking files',
            '3': 'Queued to download',
            '4': 'Downloading',
            '5': 'Queued to seed',
            '6': 'Seeding'
        }.get(status, 'Unknown (%s)' % status)

    def torrent_add(self, args):
        "torrent-add request"
        return self.make_request({
            'method': 'torrent-add',
            'arguments': args,
        })

    def torrent_set(self, args):
        "torrent-set request"
        return self.make_request({
            'method': 'torrent-set',
            'arguments': args,
        })

    def torrent_get(self, args=None):
        return self.make_request({
            'method': 'torrent-get',
            'arguments': args
        })

    def session_get(self):
        return self.make_request({
            'method': 'session-get',
            'arguments': {}
        })

    def session_stats(self):
        return self.make_request({
            'method': 'session-stats',
            'arguments': {}
        })

    def session_set(self, args):
        return self.make_request({
            'method': 'session-set',
            'arguments': args
        })

    def torrent_remove(self, ids, data=False):
        return self.make_request({
            'method': 'torrent-remove',
            'arguments': {
                'ids': ids,
                'delete-local-data': data
            }
        })

    def torrent_set_location(self, ids, directory, move=True):
        return self.make_request({
            'method': 'torrent-set-location',
            'arguments': {
                'ids': ids,
                'location': directory,
                'move': move,
            }
        })

    def blocklist_update(self):
        return self.make_request({
            'method': 'blocklist-update',
        })


def rpc_client(*args, **kwargs):
    return RPCClient(*args, **kwargs)
