'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from tropofy.widgets import Widget
from tropofy.database import DBSession
import traceback


class ExecuteFunction(Widget):
    """A button which executes a Python function when clicked.

    If your app solves a specific problem then this is where your custom code is executed.

    Below is an example of creating an ExecuteFunction widget:

    .. literalinclude:: ../../../tropofy_example_apps/core_dependencies/facility_location/facility_location.py
        :pyobject: ExecuteSolverFunction

    The following is a reference of the functions required to define an ExecuteFunction widget.
    """
    def get_type(self):
        return "ExecuteFunctionWidget"

    def refresh_from_db(self, data_set):
        return {'last_run_messages': data_set.execute_function_info.last_run_messages}

    def _set_optimisation_as_started(self, data_set):
        data_set.execute_function_info.record_execution_start()

    def update_to_db(self, data, data_set_id, oper, data_set):
        if data_set is not None:
            if oper == "prepare_to_start_function":
                self._set_optimisation_as_started(data_set)
            elif oper == "start_function":
                self._execute_function(data_set)
            elif oper == "request_progress":
                pass
            elif oper == "stopfunction":
                pass

            execute_fn_info = data_set.execute_function_info
            return {
                'startTime': execute_fn_info.last_run_start_time.strftime('%H:%M:%S %d-%m-%Y'),
                'numExecutionsForDataSet': execute_fn_info.num_times_run,
                'progressMessages': execute_fn_info.last_run_messages,
                'runningDuration': execute_fn_info.running_duration_str,
                "isCurrentlyRunning": execute_fn_info.is_currently_running,
                'summaryMessage': execute_fn_info.last_run_messages
            }
        return {}

    # Note because the developers solver function is executed as part of a SPECIFIC ajax request it is already in a thread of it's own!
    def _execute_function(self, data_set):
        try:
            self.execute_function(data_set)
        except Exception:
            data_set.send_progress_message("Error: We're sorry, something went wrong.")
            print(traceback.format_exc())
        data_set.execute_function_info.is_currently_running = False
        DBSession().flush()        

    def serialise(self):
        '''Join serialisation dict with Widget to get all params needed on client'''
        return dict(Widget.serialise(self).items() + {
            'buttonText': self.get_button_text()
        }.items())

    def get_button_text(self):
        """The text on the button in the GUI.

        :rtype: a string

        .. literalinclude:: ../../../tropofy_example_apps/core_dependencies/facility_location/facility_location.py
            :pyobject: ExecuteSolverFunction.get_button_text
            :language: python
        """
        return 'Execute Function'

    def execute_function(self, data_set):
        """Any Python code to be executed.

        :param data_set: The current data_set object through which the database is accessed.
        :type data_set: :class:`tropofy.app.AppDataSet`

        :rtype: None

        - Use :func:`tropofy.app.AppDataSet.send_progress_message` to post messages to the GUI.
        - Use :func:`tropofy.app.AppDataSet.user_is_previewing_app` to determine if a user is using your app with a preview subscription, i.e. they have not paid
        - Use :func:`tropofy.app.AppDataSet.send_preview_mode_debug_window_message` to send messages to the debug window when your app is being previewed to aid in debugging

        The following example shows the data_set being queried to determine if a user is on a paid subscription before solving

        .. literalinclude:: ../../../tropofy_example_apps/core_dependencies/facility_location/facility_location.py
            :pyobject: ExecuteSolverFunction.execute_function
            :language: python

        """
        raise NotImplementedError
