'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import openpyxl
from pyramid import threadlocal
from openpyxl.writer.excel import save_virtual_workbook
from tropofy.database import DBSession
from tropofy.database.tropofy_orm import TropofyDbMetaData
from sqlalchemy.sql import select
from pyramid.security import authenticated_userid
import datetime
import xlrd
import transaction
from operator import attrgetter


# http://pythonhosted.org/openpyxl/usage.html
# http://pythonhosted.org/openpyxl/api.html


class ExcelError(Exception):
    def __init__(self, message=''):
        self.message = message
        if type(message) is list:
            self.history_as_list = message
            self.message = '.\n'.join(message).replace('..', '.')

    def __str__(self):
        return self.message


class ClassWsMapping(object):
    """Class worksheet mapping.

    A mapping used to both write to and read from Excel. Maps a Python class to an Excel worksheet.
    """
    def __init__(self, class_, ws_name, attribute_column_aliases, process_objects=None, get_objects=None, objects=None):
        """
        :param class_: Python class in mapping.
        :type class_: class
        :param ws_name: Excel worksheet name in mapping.
        :type ws_name: str
        :param attribute_column_aliases: dict of class attribute to col alias
        :type attribute_column_aliases: dict
        :param process_objects: Function that processes the objects loaded from Excel
        :type process_objects: function
        :param get_objects: A function that accepts data_set as parameter and returns a list of objects of class_
        :type get_objects: function 
        :param objects: list of objects of class_ to be included in export. If get_objects is supplied, these objects will be overidden.
        :type objects: list
        """
        self.class_ = class_
        self.ws_name = ws_name
        self.attribute_column_aliases = attribute_column_aliases
        self.process_objects = process_objects if process_objects is not None else ClassWsMapping.process_sqla_objects
        self.get_objects = get_objects if get_objects else None
        self.objects = objects if objects else []

    def get_class(self):
        return self.class_

    @property
    def class_name(self):
        return self.class_.__name__

    @property
    def required_ws_column_names(self):
        """
        Returns all args in worksheet (including optional)

        :rtype: list
        """        
        return self.attribute_column_aliases.values()

    def get_attribute_from_column_name(self, column_name):
        """        
        :returns: Attribute as string that corresponds to column_name
        :rtype: string
        """
        for key, value in self.attribute_column_aliases.iteritems():
            if value == column_name:
                return key


    @staticmethod  # Use staticmethod as can't have cls as first attribute. Used as process_objects arg in ClassWsMapping.
    def process_sqla_objects(data_set, objects):
        DBSession().add_all(objects)
        DBSession.flush()


    @classmethod
    def create_ordered_mappings_from_sqla_classes(cls, sqla_classes=None, include_parameters_if_any=True):
        """
        :param sqla_classes: List of SQLA classes to create mappings for. If None, assumes create for all classes in app.
        :type sqla_classes: list

        :returns: list of :class:`tropofy.database.read_write_xl.ClassWsMapping`
        :rtype: list        
        """
        request = threadlocal.get_current_request()
        class_ws_mappings = []
        if request.app:
            tables_mapped_to_classes = request.app.get_tables_mapped_to_classes_for_app()
            ordered_tables = TropofyDbMetaData.get_dependency_sorted_tables([k.name for k in tables_mapped_to_classes])
            for table in ordered_tables:
                class_ = tables_mapped_to_classes.get(table)

                if sqla_classes:  # Map all classes if none specified. Also only map classes in sqla_classes
                    if class_ not in sqla_classes:
                        class_ = None

                required_attribute_names = [col.name for col in table.columns if col.name not in ['id', 'data_set_id']]
                attribute_column_aliases = {}
                for name in required_attribute_names:
                    attribute_column_aliases.update({name: name})

                if class_:
                    class_ws_mappings.append(cls(
                        class_=tables_mapped_to_classes[table],
                        ws_name=table.name,
                        attribute_column_aliases=attribute_column_aliases,
                        process_objects=cls.process_sqla_objects,
                        get_objects=lambda data_set, class_: data_set.query(class_).all(),
                    ))

        
        class ParamFromExcel(object):
            def __init__(self, name, value):
                self.name = name
                self.value = value

            @classmethod
            def load_params_into_data_set(cls, data_set, params):
                for param in params:
                    data_set.set_param(
                        name=param.name,
                        value=param.value,
                    )

            @classmethod
            def get_params_from_data_set(cls, data_set, class_):
                params = []
                for param in data_set.app.get_parameters():
                    params.append(class_(
                        name=param.name,
                        value=data_set.get_param(param.name)
                    ))
                return params

        if include_parameters_if_any and request.app.get_parameters():
            class_ws_mappings.append(cls(
                class_=ParamFromExcel,
                ws_name='parameters',
                attribute_column_aliases={
                    'name': 'name',
                    'value': 'value'
                },
                process_objects=ParamFromExcel.load_params_into_data_set,
                get_objects=ParamFromExcel.get_params_from_data_set,
            ))
        return class_ws_mappings

    def ws_has_required_columns(self, ws):  # Todo: Make it so import sheet can have extra columns. Just looks for specific column names. As is, no extra cols can exist
        headers_as_set = _ws_headers_as_set(ws)
        required_ws_column_names_as_set = set(self.required_ws_column_names)
        if required_ws_column_names_as_set <= headers_as_set:  # required_ws_column_names_as_set is a subset of headers_as_set
            return True
        raise ExcelError('The first row in a sheet must be the column names. Sheet %s is missing the following columns %s' % (self.ws_name, ", ".join(list(required_ws_column_names_as_set - headers_as_set))))

def create_excel_wb_for_tables(data_set, tables):
    class_ws_mappings = ClassWsMapping.create_ordered_mappings_from_sqla_classes(sqla_classes=tables, include_parameters_if_any=False)
    for m in class_ws_mappings:
        m.objects = data_set.query(m.get_class()).all()
        m.objects.sort(key=attrgetter(*m.required_ws_column_names))
    return create_excel_wb_from_class_mappings(data_set, ordered_class_ws_mappings=class_ws_mappings)

def create_excel_string_repr(data_set, ordered_class_ws_mappings=None):
    wb = create_excel_wb_from_class_mappings(data_set, ordered_class_ws_mappings=None)
    return save_virtual_workbook(wb)


def create_example_data_set_from_excel(data_set, file_name_with_path):  # Deprecated! Todo: Replace with load_data_from_excel_file_on_disk. Shouldn't be creating data sets in here.
    wb = xlrd.open_workbook(file_name_with_path)
    if not data_set:
        raise Exception('create_example_data_set_from_excel is deprecated. Please use load_data_from_excel_file_on_disk')
    load_data_from_excel(wb, data_set)


def create_data_set_from_excel_file_in_memory(file_in_memory):  # TodoL Replace with load_data_from_excel_file_in_memory. Shouldn't be creating data sets in here.
    wb = xlrd.open_workbook(file_contents=file_in_memory.file.read())  # openpyxl cannot read a file from a string, otherwise I would have used it
    request = threadlocal.get_current_request()
    if request.app:
        data_set = request.app.create_new_data_set_for_user(authenticated_userid(request), None, "Imported '%s'" % (file_in_memory.filename))
        return load_data_from_excel(wb, data_set)

def load_data_from_excel_file_on_disk(data_set, file_name_with_path, ordered_class_ws_mappings=None):
    wb = xlrd.open_workbook(file_name_with_path)
    load_data_from_excel(wb, data_set, ordered_class_ws_mappings)


def load_data_from_excel_file_in_memory(data_set, file_in_memory, ordered_class_ws_mappings=None):
    if not isinstance(file_in_memory, basestring):
        try:
            file_in_memory = file_in_memory.file.read()
        except:
            pass

    try:
        wb = xlrd.open_workbook(file_contents=file_in_memory)
    except:
        raise Exception('Uploaded file could not be read.')
    return load_data_from_excel(wb, data_set, ordered_class_ws_mappings)


def load_data_from_excel(wb, data_set, ordered_class_ws_mappings=None):
    """
    :param wb: Excel workbook opened with xlrd
    :param ordered_class_ws_mappings: Ordered list of :class:`tropofy.database.read_write_xl.ClassWsMapping`. Will be processed in this order (makes safe for dependencies)
    :type ordered_class_ws_mappings: list
    """
    if not ordered_class_ws_mappings:
        ordered_class_ws_mappings = ClassWsMapping.create_ordered_mappings_from_sqla_classes()

    progress_messages = []
    for mapping in ordered_class_ws_mappings:
        try:
            if mapping.ws_name in wb.sheet_names():
                ws = wb.sheet_by_name(mapping.ws_name)
                if ws and mapping.ws_has_required_columns(ws):
                    headers = ws.row_values(0)
                    new_objects = []
                    curr_row = 0
                    while curr_row < ws.nrows - 1:
                        curr_row = curr_row + 1

                        init_args = {}
                        for attr, header in mapping.attribute_column_aliases.iteritems():
                            value = _cellval(ws.row(curr_row)[headers.index(header)], wb.datemode)
                            init_args.update({attr: value})

                        new_object = None
                        try:
                            new_object = mapping.class_(**init_args)
                        except Exception as e:  # Probably a ValidationException
                            raise ExcelError([
                                "There was a problem constructing a '%s' from the data %s." % (mapping.class_name, str(init_args)),
                                str(e),
                            ])
                        try:
                            new_object.data_set_id = data_set.id
                        except AttributeError:
                            # object doesn't have a data_set_id parameter. Useful for using namedtuples as import objects that are then processed into objects on the data set in `mapping.process_objects`
                            pass
                        new_objects.append(new_object)
                    
                    try:
                        mapping.process_objects(data_set, new_objects)
                        progress_messages.extend(["Created {num} instances of type {type}".format(num=str(len(new_objects)), type=mapping.class_name)])
                    except Exception as e:
                        raise ExcelError([
                            "There was an error processing {num} instances of type {type}".format(num=str(len(new_objects)), type=mapping.class_name),
                            str(e),
                        ])
                else:
                    progress_messages.extend(["Worksheet %s does not have columns: %s." % (mapping.ws_name, mapping.required_ws_column_names - _ws_headers_as_set(ws))])
            else:
                progress_messages.extend(["There is no sheet named %s." % mapping.ws_name])
        except Exception as e:
            progress_messages.extend([
                "Aborted - Data set deleted.",
                "There was a problem with sheet <{sheet_name}>: ".format(sheet_name=mapping.ws_name),
            ])
            try:  # Get history_as_list of Excel Error if it exists
                progress_messages.extend(e.history_as_list)                    
            except AttributeError:
                progress_messages[-1] += str(e)
            
            transaction.abort()
            raise ExcelError(progress_messages)
    return progress_messages


def create_excel_wb_from_class_mappings(data_set, ordered_class_ws_mappings=None):
    """
    :param wb: Excel workbook opened with xlrd
    :param ordered_class_ws_mappings: Ordered list of :class:`tropofy.database.read_write_xl.ClassWsMapping`. Will be processed in this order (makes safe for dependencies)
    :type ordered_class_ws_mappings: list
    """
    if not ordered_class_ws_mappings:
        ordered_class_ws_mappings = ClassWsMapping.create_ordered_mappings_from_sqla_classes()
        if data_set:
            for mapping in ordered_class_ws_mappings:
                mapping.objects = mapping.get_objects(data_set, mapping.class_)

    wb = openpyxl.Workbook()
    if wb.get_sheet_by_name('Sheet') != None:
        wb.remove_sheet(wb.get_sheet_by_name('Sheet'))
    
    for mapping in ordered_class_ws_mappings:
        worksheet = wb.create_sheet()
        try:
            worksheet.title = mapping.ws_name  # The limit is 32 characters I think
        except Exception as e:
            raise e
        col_names = mapping.required_ws_column_names
        if col_names:
            for col in col_names:
                worksheet.cell(row=0, column=col_names.index(col)).value = col
            
            for i, obj in enumerate(mapping.objects):
                for j, col_name in enumerate(col_names):
                    try:
                        value = getattr(obj, mapping.get_attribute_from_column_name(col_name))
                    except AttributeError:
                        print('Object in ClassWsMappingForExport.objects does not match ClassWsMappingForExport object and col definition')
                        raise
                    worksheet.cell(row=i + 1, column=j).value = value  # Add one to row to skip col names header row.
    return wb


def _get_column_for_table_ignoring_id_etc(table):
    return [col.name for col in table.columns if col.name not in ['id', 'data_set_id']]


def _cellval(cell, datemode):
    if cell.ctype == xlrd.XL_CELL_DATE:
        datetuple = xlrd.xldate_as_tuple(cell.value, datemode)
        if datetuple[:3] == (0, 0, 0):
            return datetime.time(datetuple[3], datetuple[4], datetuple[5])
        #elif datetuple[3:] == (0, 0, 0):  # Doesn't work for 12:00 am!
        #    return datetime.date(datetuple[0], datetuple[1], datetuple[2])
        else:
            return datetime.datetime(datetuple[0], datetuple[1], datetuple[2], datetuple[3], datetuple[4], datetuple[5])
    if cell.ctype == xlrd.XL_CELL_EMPTY:
        return None
    if cell.ctype == xlrd.XL_CELL_BOOLEAN:
        return cell.value == 1
    return cell.value


def _get_table_rows_as_tuple_list(engine, table, data_set_id):  # Todo: Remove from this file
    query = select([table]).where(table.c.data_set_id == data_set_id) if data_set_id is not None and hasattr(table.c, 'data_set_id') else select([table])
    result = engine.connect().execute(query)
    return result


def _create_excel_workbook_for_tables_with_optional_data(tables, engine, data_set_id=None):  # Note this is not SQLAlchemy ORM driven():
    wb = openpyxl.Workbook()
    if wb.get_sheet_by_name('Sheet') != None:
        wb.remove_sheet(wb.get_sheet_by_name('Sheet'))
    ordered_tables = TropofyDbMetaData.get_dependency_sorted_tables([k.name for k in tables])
    for table in ordered_tables:
        worksheet = wb.create_sheet()
        worksheet.title = table.name
        col_names = [col.name for col in table.columns if col.name not in ['id', 'data_set_id']]
        all_col_names = [col.name for col in table.columns]
        if col_names:
            for col in col_names:
                worksheet.cell(row=0, column=col_names.index(col)).value = col
            data = [list(row) for row in _get_table_rows_as_tuple_list(engine, table, data_set_id)] if data_set_id else []
            for row in data:
                col_counter_for_writing = 0
                col_counter = 0
                for col in row:
                    if all_col_names[col_counter] not in ['id', 'data_set_id']:
                        worksheet.cell(row=data.index(row) + 1, column=col_counter_for_writing).value = col  # +1 to not stomp on the header row
                        col_counter_for_writing = col_counter_for_writing + 1
                    col_counter = col_counter + 1  # beware list.index method for repeated values!
    return wb

# ws helper function
def _ws_headers_as_set(ws):
    headers = ws.row_values(0)
    return set([col.encode('ascii', 'ignore') for col in headers if col != None])