'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from tropofy.widgets import Widget, FileUpload, Form
from tropofy.database import DBSession, read_write_xl
from tropofy.app import AppDataSet
from pyramid import threadlocal
from pyramid.security import authenticated_userid
from sqlalchemy.orm.exc import MultipleResultsFound, NoResultFound
from sqlalchemy.sql.expression import desc


class DataSets(Widget):
    # all_actions: selectDataSet, checkNumberOfDataSets
    # all_events: beforeDisplayImportOption, beforeDisplayOpenOption

    def __init__(self):
        # New
        self.add_embedded_widget(name='newDataSetForm', widget=NewDataSetForm())

        # Open
        open_data_set_form = SelectExistingDataSet()
        self.add_embedded_widget(name='openDataSetSelectForm', widget=open_data_set_form)
        
        # Import
        import_new_ds_upload_widget = ImportNewDataSetFileUpload()        
        import_ds_name_edit = ImportDataSetNameEdit()
        import_new_ds_upload_widget.add_event_subscription('fileUploadSuccess', [
            self.actions('selectDataSet'),
            import_ds_name_edit.actions('show'),
        ])
        self.add_event_subscription('beforeDisplayImportOption', [
            import_new_ds_upload_widget.actions('refresh'),
            import_ds_name_edit.actions('hide')
        ])
        self.add_embedded_widget(name='importNewDataSetFileUpload', widget=import_new_ds_upload_widget)
        self.add_embedded_widget(name='importDataSetNameEdit', widget=import_ds_name_edit)

        # Export
        export_data_set_form = SelectExistingDataSet()
        self.add_embedded_widget(name='exportDataSetSelectForm', widget=export_data_set_form)

        # Manage
        manage_ds_form = ManageDataSetForm()
        self.add_embedded_widget(name='manageDsForm', widget=manage_ds_form)

        # Managing data set changes, refreshing data set select forms.

        refresh_actions_on_data_set_changes = [
            manage_ds_form.actions('refresh'),
            open_data_set_form.actions('refresh'),
            export_data_set_form.actions('refresh'),
        ]

        manage_ds_form.add_event_subscription('formSubmitSuccess', [  # Exclude manage_ds_form - manually refreshed in js.
            open_data_set_form.actions('refresh'),
            export_data_set_form.actions('refresh'),
        ])
        import_new_ds_upload_widget.add_event_subscription('fileUploadSuccess', refresh_actions_on_data_set_changes)
        import_ds_name_edit.add_event_subscription('formSubmitSuccess', refresh_actions_on_data_set_changes)

        manage_ds_form.add_event_subscription('refreshed', self.actions('checkNumberOfDataSets'))

        
    def get_type(self):
        return "DataSetsWidget"


class NewDataSetForm(Form):
    def get_form_elements(self, data_set):        
        app = threadlocal.get_current_request().app

        options = [name for name in app.get_example_app_names()]
        options.append('None')

        default = app.get_default_example_data_set_name()

        return [
            Form.Element(
                name='data_set_template',
                input_type='select',
                default=default if default and default in options else options[0],
                label='Template',
                options=options,
            ),
            Form.Element(
                name='data_set_name',
                input_type='text',
                label='Data Set Name',
                default='New data set',
            ),
        ]

            
    def process_data(self, data, data_set):
        pass

    def is_autosubmit(self, data_set):
        return False

    def hide_submit_button(self, data_set):
        return True


class ImportNewDataSetFileUpload(FileUpload):  # Warning: get_allowed_file_types below is referenced as an example in the docs.
    def get_allowed_file_types(self, data_set):
        return [
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'application/vnd.ms-excel.sheet.macroEnabled.12',
            'application/vnd.ms-excel',
            'application/x-zip-compressed',
            'application/octet-stream',
        ]

    def process_file(self, data_set, file_in_memory, result):
        request = threadlocal.get_current_request()
        data_set = request.app.create_new_data_set_for_user(authenticated_userid(request), None, "Imported '%s'" % (file_in_memory.filename))
        result['dataSetId'] = data_set.id
        result['dataSetName'] = data_set.name
        result['message'] = read_write_xl.load_data_from_excel_file_in_memory(data_set, file_in_memory)


class ImportDataSetNameEdit(Form):
    def get_form_elements(self, data_set):
        user_email = authenticated_userid(threadlocal.get_current_request())
        app_name = threadlocal.get_current_request().app.name
        if user_email and app_name:
            data_set = DBSession().query(AppDataSet).filter_by(user_email=user_email).filter_by(app_name=app_name).order_by(desc(AppDataSet.id)).first()

            return [
                Form.Element(
                    name='data_set_id',
                    input_type='text',
                    default=data_set.id,
                    hidden=True,
                ),
                Form.Element(
                    name='data_set_name',
                    input_type='text',
                    label='Imported Data Set Name',
                    default=data_set.name,
                ),
            ]

    def process_data(self, data, data_set):
        try:
            data_set = DBSession().query(AppDataSet).filter_by(id=data['data_set_id']).one()
            data_set.name = data['data_set_name']
        except (NoResultFound, MultipleResultsFound):
            pass

    def autosubmit_entire_form(self, data_set):
        return True

    def is_hidden(self):
        return True


class SelectExistingDataSet(Form):
    def get_form_elements(self, data_set):
        data_set_select_form_element = get_existing_data_sets_form_select(data_set)
        return [data_set_select_form_element] if data_set_select_form_element else []

    def is_autosubmit(self, data_set):
        return False

    def hide_submit_button(self, data_set):
        return True

    def process_data(self, data, data_set):
        pass


class DataSetDateTimeAccessInfo(Form):
    def get_form_elements(self, data_set):
        return [
            Form.Element(
                name='created_datetime',
                input_type='datetime',
                label='Created',
                default='',
                disabled=True,
            ),
            Form.Element(
                name='last_accessed_datetime',
                input_type='datetime',
                label='Last accessed',
                default='',
                disabled=True,
            ),
        ]

    def process_data(self, data, data_set):
        pass

    def is_autosubmit(self, data_set):
        return False

    def hide_submit_button(self, data_set):
        return True


class ManageDataSetForm(Form):
    def get_form_elements(self, data_set):
        data_set_select_form_element = get_existing_data_sets_form_select(data_set)
        if data_set_select_form_element:
            return [
                data_set_select_form_element,
                Form.Element(
                    name='new_data_set_name',
                    input_type='text',
                    label='New data set name',
                    default='',
                ),
                Form.Element(
                    name='delete_data_set',
                    input_type='text',
                    label='',
                    default=False,
                    hidden=True
                ),
            ]
        else:
            return []

    def process_data(self, data, data_set):
        try:
            data_set = DBSession().query(AppDataSet).filter_by(id=data['data_set[dataSetId]']).one()

            if data['delete_data_set'] == 'true':  # Form doesn't deal with Boolean very well yet so have to check against string.
                DBSession().delete(data_set)
            else:
                if data['new_data_set_name']:
                    data_set.name = data['new_data_set_name']
                else:
                    return {
                        'success': False,
                        'message': '',
                        'results': [{
                            'name': 'new_data_set_name', 'success': False, 'message': 'Data set name cannot be empty.'
                        }]
                    }
        except (NoResultFound, MultipleResultsFound):
            # Data set id on GUI does not exist on server. 
            # If multiple people signed in as same user, and one deletes and app while another has manage app dialog open, this could happen.
            pass

    def is_autosubmit(self, data_set):
        return False

    def hide_submit_button(self, data_set):
        return True


def get_existing_data_sets_form_select(data_set):
    user_email = authenticated_userid(threadlocal.get_current_request())
    app_name = threadlocal.get_current_request().app.name
    if user_email and app_name:
        data_sets = DBSession().query(AppDataSet).filter_by(user_email=user_email, app_name=app_name).order_by(desc(AppDataSet.id)).all()

        options = []
        for data_set in data_sets:
            options.append({
                'value': {
                    'dataSetId': data_set.id,
                    'dataSetName': data_set.name,
                    'createdDateTime': data_set.created_datetime,
                    'lastAccessedDateTime': data_set.last_accessed_datetime,
                },
                'text': data_set.name
            })

        if options:
            return Form.Element(
                name='data_set',
                input_type='select',
                default=options[0]['value'],
                label='Data Set Name',
                options=options,
                disabled=len(options) == 0
            )
