from sqlalchemy.types import Text, Float, Integer
from sqlalchemy.schema import Column, ForeignKeyConstraint, UniqueConstraint
from tropofy.database.tropofy_orm import DataSetMixin
from tropofy.app import AppWithDataSets, Step, StepGroup
from tropofy.widgets import SimpleGrid, KMLMap, Chart
from simplekml import Kml


class Store(DataSetMixin):
    name = Column(Text, nullable=False)
    latitude = Column(Float, nullable=False)
    longitude = Column(Float, nullable=False)

    __table_args__ = (UniqueConstraint('name', 'data_set_id'),)


class Performance(DataSetMixin):
    store_name = Column(Text, nullable=False)
    year = Column(Integer, nullable=False)
    sales = Column(Float, nullable=False, default=0)
    expenses = Column(Float, nullable=False, default=0)

    __table_args__ = (
        UniqueConstraint('store_name', 'year', 'data_set_id'),
        ForeignKeyConstraint(['store_name', 'data_set_id'], ['store.name', 'store.data_set_id'], ondelete='CASCADE', onupdate='CASCADE'),
    )


class MyKMLMap(KMLMap):
    def get_kml(self, data_set):
        kml = Kml()
        for store in data_set.query(Store).all():
            kml.newpoint(name=store.name, coords=[(store.longitude, store.latitude)])
        return kml.kml()


class PerformanceBarChart(Chart):
    def get_chart_type(self, data_set):
        return Chart.BARCHART

    def get_table_schema(self, data_set):
        return {
            "year": ("string", "Year"),
            "sales": ("number", "Sales"),
            "expenses": ("number", "Expenses")
        }

    def get_table_data(self, data_set):
        results = []
        years = [y for r in data_set.query(Performance.year).distinct() for y in r]
        for year in years:
            performances = data_set.query(Performance).filter_by(year=year).all()
            results.append({
                "year": year,
                "sales": sum(p.sales for p in performances),
                "expenses": sum(p.expenses for p in performances),
            })
        return results

    def get_column_ordering(self, data_set):
        return ["year", "sales", "expenses"]

    def get_order_by_column(self, data_set):
        return "year"

    def get_chart_options(self, data_set):
        return {
            'title': 'Company Performance',
            'vAxis': {
                'title': 'Year',
                'titleTextStyle': {'color': 'red'}
            }
        }


class MyFirstApp(AppWithDataSets):
    def get_name(self):
        return "My First App"

    def get_gui(self):
        step_group_1 = StepGroup(name='Input')
        step_group_1.add_step(Step(name='Stores', widgets=[SimpleGrid(Store)]))
        step_group_1.add_step(Step(name='Performances', widgets=[SimpleGrid(Performance)]))

        step_group_2 = StepGroup(name='Output')
        step_group_2.add_step(Step(name='Map', widgets=[MyKMLMap()]))
        step_group_2.add_step(Step(name='Chart', widgets=[PerformanceBarChart()]))

        return [step_group_1, step_group_2]
