'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from tropofy.database.db_manager import DbManager
from tropofy.widgets import GridWidget
from datetime import date, time, datetime

PYTHON_TYPE_TO_FORM_TYPE = {  # SND: Remove this! Duplicated functionality in FormWidget
    int: 'text',
    str: 'text',
    float: 'text',
    bool: 'select',
    date: 'date',
    time: 'time',
    datetime: 'datetime',
}


class SimpleGrid(GridWidget):
    """Basic Grid. This widget is special. It is not used like the other widgets. You do not need to create a class
    which implements function to use an instance of this widget

    :param source_class: A user defined SQLAlchemy Python class

    .. note::
        The source_class provided to a SimpleGrid must have the following properties:
            - The database column names must be the same as the member variable names, i.e. you cannot use code like ``some_name = Column('some_other_name', String(50))``),
            - The initialisers parameter names must be the same as the member variable names. Note that SQLAlchemy provides a default initialiser with keyword arguements equal to the names of the column members defined for your class.

    The following class satisfies the above requirements

    .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
        :pyobject: ConformingClass
    """
    def __init__(self, source_class):
        self.source_class = source_class

    def delete_row(self, obj_id):
        DbManager.delete_object_from_db(source_class=self.source_class, obj_id=obj_id)

    def get_table_hierachy_from_inheritance_chain(self):
        full_class_hierachy = list(self.source_class.__bases__) + [self.source_class]
        return [class_type for class_type in full_class_hierachy if hasattr(class_type, "__table__")]

    def add_new_row(self, data, data_set_id):
        edited_obj = DbManager.construct_object_and_add_to_db(source_class=self.source_class, data=data, data_set_id=data_set_id)
        return edited_obj.as_json_data_row(self.get_grids_column_names())

    def get_grids_column_names(self):
        return [col.name for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns]

    def get_column_form_input_types(self):
        return [PYTHON_TYPE_TO_FORM_TYPE.get(col.type.python_type) for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns]

    def edit_row(self, data):
        edited_obj = DbManager.update_object_in_db(source_class=self.source_class, data=data)
        return edited_obj.as_json_data_row(self.get_grids_column_names())

    def get_refresh_objects(self, data_set):
        if hasattr(self.source_class, 'data_set_id') and data_set:
            return data_set.query(self.source_class).all()
        return []  # If an app could not filter by data_set_id - it would go here!

    def get_type(self):
        return 'SimpleGrid'

    def grid_is_editable(self):
        return True
