'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import imp
import os
import inspect

from tropofy.app import LoadedApps, App, AppDataSet, DataSetVar, SavedImageReference, FunctionExecutionInfo
from tropofy.database.tropofy_orm import TropofyDbMetaData


class AppManager(object):

    @staticmethod
    def get_app_from_url_name(url_name):
        return LoadedApps.apps.get(url_name)

    @staticmethod
    def get_app_if_only_one():
        if len(LoadedApps.apps) == 1:
            return AppManager.get_app_from_url_name(LoadedApps.apps.keys()[0])
        return None

    @classmethod
    def initialise_apps(cls, apps_dir):
        dirs = apps_dir.split(";")
        for dir_ in dirs:
            possible_app_modules = []
            cls.import_all_py_files_under_directory(module_name='app', file_location=dir_, modules=possible_app_modules)

            for module_def in possible_app_modules:
                app = AppManager._create_app(module_def)
                if app and app.url_name != 'user_management':
                    TropofyDbMetaData.metadata.create_all(
                        bind=app.engine,
                        tables=app.get_orm_tables() + [
                            TropofyDbMetaData.metadata.tables[AppDataSet.__tablename__],
                            TropofyDbMetaData.metadata.tables[DataSetVar.__tablename__],
                            TropofyDbMetaData.metadata.tables[SavedImageReference.__tablename__],
                            TropofyDbMetaData.metadata.tables[FunctionExecutionInfo.__tablename__],                            
                        ]
                    )
            if len(LoadedApps.apps) == 0:
                raise Exception('No apps found in ' + dir_ + '.\n\nLikely issues:\n- tropofy_run was not executed from the intended folder.\n- No class inherits from AppWithDataSets in your app.')

    @classmethod
    def import_all_py_files_under_directory(cls, module_name, file_location, modules):
        # Beware of cases when there is a file called my_app.py and a directory called my_app
        #return [name for name in os.listdir(dir)
        #    if os.path.isdir(os.path.join(dir, name))]

        if not (file_location.endswith('tropofy_example_apps/tutorials') or file_location.endswith('setup.py')):  # special catch all for the tutorials subdirectory and any 'setup.py' files
            if file_location.endswith('.py'):
                modules.append(imp.load_source(module_name.strip('.py'), file_location))
            elif os.path.isdir(file_location):
                for sub_module_name in [name for name in os.listdir(file_location)]:
                    cls.import_all_py_files_under_directory(
                        module_name=module_name + '_' + sub_module_name,
                        file_location=file_location + '/' + sub_module_name,
                        modules=modules
                    )

    @staticmethod
    def _create_app(imported_module_definition, preview_name_string=''):
        app = AppManager._get_app_interface_from_module(imported_module_definition)
        if app:  # module_definition may not have a class that inherits from App.
            LoadedApps.apps.pop(app.url_name, None)
            LoadedApps.apps[app.url_name] = app
            return LoadedApps.apps[app.url_name]

    @staticmethod
    def _get_app_interface_from_module(imported_module_definition):
        ret_val = None
        for _, class_type in inspect.getmembers(imported_module_definition, inspect.isclass):
            if class_type.__module__ == str(imported_module_definition.__name__):
                if App in inspect.getmro(class_type):
                    if ret_val is not None:
                        return None
                    ret_val = class_type()
        return ret_val
