'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import requests
import json
from simplekml import Kml, Style, IconStyle, Icon
from sqlalchemy.schema import Column
from sqlalchemy.types import Text, Float

from tropofy.app import AppWithDataSets, Step, StepGroup
from tropofy.widgets import ExecuteFunction, KMLMap, SimpleGrid
from tropofy.database.tropofy_orm import DataSetMixin


class InputAddress(DataSetMixin):
    street = Column(Text, nullable=False)
    suburb = Column(Text, nullable=False, default='')
    city = Column(Text, nullable=False, default='')
    state = Column(Text, nullable=False, default='')
    post_code = Column(Text, nullable=False, default='')
    country = Column(Text, nullable=False, default='')

    def __init__(self, street, suburb='', city='', state='', post_code='', country=''):
        self.street = street
        self.suburb = suburb
        self.city = city
        self.state = state
        self.post_code = post_code
        self.country = country


class OutputGeocodedLocation(DataSetMixin):
    address = Column(Text, nullable=False)
    latitude = Column(Float, nullable=True)
    longitude = Column(Float, nullable=True)
    city = Column(Text, nullable=True)
    state = Column(Text, nullable=True)
    country = Column(Text, nullable=True)
    post_code = Column(Text, nullable=True)
    geocode_quality = Column(Text, nullable=True)

    def __init__(self, address, latitude=None, longitude=None, state=None, city=None, country=None, post_code=None, geocode_quality=None):
        self.address = address
        self.latitude = latitude
        self.longitude = longitude
        self.city = city
        self.state = state
        self.country = country
        self.post_code = post_code
        self.geocode_quality = geocode_quality


class GeocodeAddresses(ExecuteFunction):

    def get_button_text(self):
        return "Geocode your Addresses"

    def execute_function(self, data_set):

        if len(data_set.query(InputAddress).all()) > 20:
            data_set.send_progress_message("You can only geocode up to 20 locations using the free version of this app")
        else:
            data_set.send_progress_message("Deleting old results")
            data_set.query(OutputGeocodedLocation).delete()

            data_set.send_progress_message("Geocoding addresses")
            result_text = call_geocoding_api(data_set)

            if result_text:
                data_set.send_progress_message("Writing results to DB")
                results = json.loads(remove_non_ascii(result_text))
                gc = [construct_gl_from_result(mq_provided_location_as_string(result['providedLocation']), result['locations'][0] if result['locations'] else None) for result in results['results']]
                data_set.add_all(gc)

            data_set.send_progress_message("Finished")


class KMLMapOutput(KMLMap):

    def get_kml(self, data_set):
        kml = Kml()
        mystyle = Style(iconstyle=IconStyle(scale=0.8, icon=Icon(href='http://maps.google.com/mapfiles/kml/paddle/blu-circle-lv.png')))
        for gc in data_set.query(OutputGeocodedLocation).all():
            if gc.longitude and gc.latitude:
                point = kml.newpoint(name=gc.address, coords=[(gc.longitude, gc.latitude)])
                point.style = mystyle
        return kml.kml()


class MyBulkGeocoderApp(AppWithDataSets):

    def get_name(self):
        return "Batch Geocoding"

    def get_examples(self):
        return {"Demo Brisbane Addresses": self.load_example_data_for_brisbane,
                "Demo New York Addresses": self.load_example_data_for_new_york}

    def get_gui(self):
        step_group1 = StepGroup(name='Enter your data')
        step_group1.add_step(Step(
            name='Enter addresses data',
            widgets=[SimpleGrid(InputAddress)],
            help_text="""
                You can can enter the entire address into the 'street' column, or if you have the address information broken up you can
                enter the components of each address into the right columns, this will generally give you higher quality answers"""
        ))

        step_group2 = StepGroup(name='Geocode')
        step_group2.add_step(Step(name='Geocode your Addresses', widgets=[GeocodeAddresses()]))

        step_group3 = StepGroup(name='View Geocodes')
        step_group3.add_step(Step(
            name='View Geocodes',
            widgets=[SimpleGrid(OutputGeocodedLocation), KMLMapOutput()],
            help_text='''
                The grid lists the latitude and longitude of each address that was geocoded.
                The map displays a preview of the KML (Google Earth) file that you can download below'''
        ))

        return [step_group1, step_group2, step_group3]

    @staticmethod
    def load_example_data_for_brisbane(data_set):
        addresses = []
        addresses.append(InputAddress("6 Luke St, Wavell Heights, Brisbane, 4012"))
        addresses.append(InputAddress("244 Edinburgh Castle Rd, Wavell Heights, Brisbane, 4012"))
        addresses.append(InputAddress("4 Aloe St, Wavell Heights, Brisbane, 4012"))
        addresses.append(InputAddress("6 Sylvan Ave", "Wavell Heights", "Brisbane", "QLD", "4012", "Australia"))
        addresses.append(InputAddress("162 Shaw Rd", "Wavell Heights", "Brisbane", "QLD", "4012", "Australia"))
        data_set.add_all(addresses)

    @staticmethod
    def load_example_data_for_new_york(data_set):
        addresses = []
        addresses.append(InputAddress("Worth st, Lower Manhatten, New York"))
        addresses.append(InputAddress("Fulton st, Lower Manhatten, New York"))
        addresses.append(InputAddress("Washington St", "MeatPacking District", "New York City", "New York", "", "USA"))
        data_set.add_all(addresses)

    def get_icon_url(self):
        return 'http://www.tropofy.com/static/css/img/tropofy_example_app_icons/geocoding.png'

    def get_home_page_content(self):
        return {
            'content_app_name_header': '''
            <div>
            <span style="vertical-align: middle;">Batch Geocoding</span>
            <img src="http://www.tropofy.com/static/css/img/tropofy_example_app_icons/geocoding.png" alt="main logo" style="width:15%">
            </div>''',

            'content_single_column_app_description': '''
            Ever wanted to:
            <ul>
            <li>Determine the latitude and longitude for a bunch of locations from their addresses anywhere in the world?</li>
            <li>View the geocoded locations on a map?</li>
            <li>Download a KML file showing the geocoded locations?</li>
            <li>Use a simple interface with csv importing of input and exporting of output data?</li>
            </ul>
                <p>This app might be what you are looking for! Sign up and give it a go.</b></p>
            <p>Need help or wish this app had one more feature, contact us at <b>info@tropofy.com</b> to see if we can help</p>''',

            'content_double_column_app_description_1': '',
            'content_double_column_app_description_2': '',
            'content_row_2_col_1_header': '',
            'content_row_2_col_1_content': '',
            'content_row_2_col_2_header': '',
            'content_row_2_col_2_content': '',
            'content_row_2_col_3_header': '',
            'content_row_2_col_3_content': '',

            'content_row_3_col_1_header': 'Input Data',

            'content_row_3_col_1_content': '''
            <p>Simply enter a list of locations defined by their addresses</p>
            <p>Addresses can be broken up into their different components ie. street, suburb, city, state, post code and country, or simple entered as a single string</p>
            <p>Entering the address info broken up into its different components will enable to geocoder to return higher quality answers</p>''',

            'content_row_3_col_2_header': 'Output Data',

            'content_row_3_col_2_content': '''
            <p>For each location geocoded, the latitude, longitude as well as the matched city, state, country and post code of the latitude and longitude are returned.
            A quality measure of the result is also returned.</p>
            <p>You can view the results on a map inside the app, and download a KML to view the solution using Google Earth.</p>''',

            'content_row_4_col_1_header': "Tropofy",

            'content_row_4_col_1_content': '''
            This app was created using the <a href="http://www.tropofy.com" target="_blank">Tropofy platform</a>.
            <div style="font-size:70%">
            <p>Geocoding Courtesy of <a href="http://www.mapquest.com/" target="_blank">MapQuest</a> <img src="http://developer.mapquest.com/content/osm/mq_logo.png">.
            Data courtesy of OpenStreetMap contributors. <a href="www.openstreetmap.org/copyright or www.opendatacommons.org/licenses/odbl" target="_blank">License<a></p>
            </div>'''
        }


def call_geocoding_api(data_set):
    '''
    For some mapquest development resources see...
    http://developer.mapquest.com/web/info/terms-of-use
    http://www.mapquestapi.com/geocoding/#batch
    http://developer.mapquest.com/web/products/open/geocoding-service
    '''
    addresses_to_geocode = {'locations': [{'street': b.street, 'city': b.city, 'state': b.state, 'postalCode': b.post_code, 'country': b.country} for b in data_set.query(InputAddress).all()]}

    key = 'Fmjtd%7Cluub2d012u%2Cra%3Do5-9ua2l6'
    url = 'http://www.mapquestapi.com/geocoding/v1/batch?key=%s' % (key)

    try:
        response = requests.post(url, data=json.dumps(addresses_to_geocode))  # Send json in POST body
        if not response.ok:
            if response.status_code == 404:
                data_set.send_progress_message("An error occurred connecting to http://www.mapquestapi.com.")
            elif response.status_code == 500:
                data_set.send_progress_message("The server couldn't fulfill the request.")                
            else:
                data_set.send_progress_message('An error occurred')
            data_set.send_progress_message(response.content)
            return ''
    except requests.exceptions.ConnectionError:
        data_set.send_progress_message("An error occurred connecting to http://www.mapquestapi.com.")
        return ''
    return response.content


def remove_non_ascii(s):
    return "".join(i for i in s if ord(i) < 128)


def mq_provided_location_as_string(data):
    return ", ".join(value for key, value in data.items() if value)


def interpret_mq_geocode_quality(qual):
    if qual[:2] in ['P1', 'L1', 'I1', 'B1', 'B2', 'B3']:
        return "Excellent"
    elif qual[:2] in ['B2', 'B3']:
        return "Fair"
    else:
        return "Poor"


def construct_gl_from_result(address, result):
    if result is not None:
        return OutputGeocodedLocation(
            address=address,
            latitude=result['displayLatLng']['lat'],
            longitude=result['displayLatLng']['lng'],
            state=result['adminArea3'],
            city=result['adminArea5'],
            country=result['adminArea1'],
            post_code=result['postalCode'],
            geocode_quality=interpret_mq_geocode_quality(result['geocodeQualityCode'])
        )
    return OutputGeocodedLocation(address)
