'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import imp
import os
import inspect

from tropofy.app import LoadedApps, App, AppDataSet, DataSetVar, SavedImageReference, FunctionExecutionInfo
from tropofy.database.tropofy_orm import TropofyDbMetaData, DynamicSchemaName
from sqlalchemy.engine import reflection
from sqlalchemy.schema import CreateSchema
from sqlalchemy import create_engine


class AppManager(object):

    @staticmethod
    def get_app_from_url_name(url_name):
        return LoadedApps.apps.get(url_name)

    @staticmethod
    def get_app_if_only_one():
        if len(LoadedApps.apps) == 1:
            return AppManager.get_app_from_url_name(LoadedApps.apps.keys()[0])
        return None

    @classmethod
    def initialise_apps(cls, apps_dir):
        if DynamicSchemaName.use_app_schemas:
            cls._create_schema_if_required(create_engine(App.overridden_global_sqla_engine), DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME)
        
        dirs = apps_dir.split(";")
        for dir_ in dirs:

            possible_apps = []
            cls.import_all_py_files_under_directory(module_name='app', file_location=dir_, possible_apps=possible_apps)

            for possible_app in possible_apps:
                app = AppManager._create_app(possible_app)
                if app and app.url_name != 'user_management':
                    # Create schema if need to.
                    cls._create_schema_if_required(app.engine, possible_app.schema_name)

                    TropofyDbMetaData.metadata.create_all(
                        bind=app.engine,
                        tables=app.get_orm_tables() + [
                            TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(AppDataSet.__tablename__)],
                            TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(DataSetVar.__tablename__)],
                            TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(SavedImageReference.__tablename__)],
                            TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(FunctionExecutionInfo.__tablename__)],
                        ]
                    )
            if len(LoadedApps.apps) == 0:
                raise Exception('No apps found in ' + dir_ + '.\n\nLikely issues:\n- tropofy_run was not executed from the intended folder.\n- No class inherits from AppWithDataSets in your app.')

    @classmethod
    def import_all_py_files_under_directory(cls, module_name, file_location, possible_apps):
        # Beware of cases when there is a file called my_app.py and a directory called my_app
        #return [name for name in os.listdir(dir)
        #    if os.path.isdir(os.path.join(dir, name))]

        if not (file_location.endswith('tropofy_example_apps/tutorials') or file_location.endswith('setup.py')):  # special catch all for the tutorials subdirectory and any 'setup.py' files
            if file_location.endswith('.py'):
                DynamicSchemaName.set_schema_name(os.path.basename(file_location).strip('.py'))
                possible_apps.append(PossibleApp(
                    module=imp.load_source(module_name.strip('.py'), file_location),
                    schema_name=DynamicSchemaName.schema_name, 
                ))
            elif os.path.isdir(file_location):
                for sub_module_name in [name for name in os.listdir(file_location)]:
                    cls.import_all_py_files_under_directory(
                        module_name=module_name + '_' + sub_module_name,
                        file_location=file_location + '/' + sub_module_name,
                        possible_apps=possible_apps
                    )

    @staticmethod
    def _create_app(possible_app, preview_name_string=''):
        app = AppManager._get_app_interface_from_module(possible_app.module)
        if app:  # module_definition may not have a class that inherits from App.
            app.schema_name = possible_app.schema_name
            LoadedApps.apps.pop(app.url_name, None)
            LoadedApps.apps[app.url_name] = app
            return LoadedApps.apps[app.url_name]

    @staticmethod
    def _get_app_interface_from_module(imported_module_definition):
        ret_val = None
        for _, class_type in inspect.getmembers(imported_module_definition, inspect.isclass):
            if class_type.__module__ == str(imported_module_definition.__name__):
                if App in inspect.getmro(class_type):
                    if ret_val is not None:
                        return None
                    ret_val = class_type()
        return ret_val

    @classmethod
    def _create_schema_if_required(cls, engine, schema_name):
        """Schema creation is required if using app schemas, and the schema does not exist yet."""
        if schema_name and not cls._schema_exist(engine, schema_name):
            engine.connect().execute(CreateSchema(schema_name))

    @classmethod
    def _schema_exist(cls, engine, schema_name):
        insp = reflection.Inspector.from_engine(engine)
        return schema_name in insp.get_schema_names()


class PossibleApp(object):
    def __init__(self, module, schema_name):
        self.module = module
        self.schema_name = schema_name