'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''
from collections import namedtuple, defaultdict


class Widget(object):
    """A widget is a HTML interface to the database via a developer defined query or framework specified query"""
    widget_indexer = 0
    widget_dict = {}

    Action = namedtuple('Action', 'widget name')
    # all_actions = ['refresh', 'hide', 'show']

    def actions(self, name):
        """Return an event object to pass to add_event_subscription"""
        return Widget.Action(self, name)

    def add_event_subscription(self, trigger_event_name, subscribed_actions):
        """Add subscription(s) to a trigger event on this widget.

        :param trigger_event_name: 
        :param subscribed_actions: 
        :type subscribed_actions: a Widget.Action or list of Widget.Action's.
        """
        if not hasattr(self, 'event_subscriptions'):  # Create dict: event_name: [list of subscriptions]
            self.event_subscriptions = defaultdict(list)
        if type(subscribed_actions) is not list:
            subscribed_actions = [subscribed_actions]
        # Add checking that trigger_event_name is a valid event. Perhaps with an enum pattern?
        self.event_subscriptions[trigger_event_name] += subscribed_actions

    def allocate_id(self):
        self.id = Widget.widget_indexer
        Widget.widget_indexer += 1
        assert self.id not in Widget.widget_dict
        Widget.widget_dict[self.id] = self
        self.num_cols_in_gui = 12

    def get_unique_div_id(self):
        return "widget-" + str(self.id)

    def get_type(self):
        '''Unique identifier for the widget'''
        pass

    def get_info_to_download(self, data_set, as_attachment):
        return ''

    def update_to_db(self, data, data_set_id, oper='', data_set=None):
        """Update some data from the widget to the db."""
        pass

    def update_from_db(self, data, data_set_id, oper='', data_set=None):
        """Update some data from the server to the widget."""
        pass

    def refresh_from_db(self, data_set, request):
        """Refresh all data for widget."""
        pass

    def add_embedded_widget(self, name, widget):
        
        if not hasattr(self, 'embedded_widgets'):
            self.embedded_widgets = {}

        widget.allocate_id()
        self.embedded_widgets.update({name: widget})

    __mapper_args__ = {  # TODO: Remove these? No longer using SQLA here....
        'polymorphic_identity': 'widget',
        'polymorphic_on': type
    }

    def is_hidden(self):  # Todo: Could be changed to include data_set and added to all widget.refresh_from_db. Would require standardisation of refresh_from_db though.
        """Hides a widget. Can be shown through events. Default: False.

        Warning: this is evaluated once on app launch. After this, visibility must be managed by events.
        :returns: bool
        """
        return False

    def serialise(self):
        '''Dict representation for easy conversion to JSON.'''        
        event_subscriptions = self.event_subscriptions if hasattr(self, 'event_subscriptions') else {}
        serialised_event_subscriptions = {}
        for event_name, subscribed_actions in event_subscriptions.iteritems():            
            serialised_event_subscriptions.update({event_name: []})
            for action in subscribed_actions:
                serialised_event_subscriptions[event_name].append({
                    'widget_id': action.widget.id,
                    'name': action.name
                })

        return {
            'id': self.id,
            'type': self.get_type(),
            'numColsInGui': self.num_cols_in_gui,
            'embeddedWidgetSpecs': {key: value.serialise() for key, value in self.embedded_widgets.iteritems()} if hasattr(self, 'embedded_widgets') else {},
            'eventSubscriptions': serialised_event_subscriptions,
            'isHidden': self.is_hidden()
        }
