'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from math import radians, cos, sin, asin, sqrt
from sqlalchemy.types import Float, Boolean, Text
from sqlalchemy.schema import Column, ForeignKeyConstraint, UniqueConstraint
from sqlalchemy.orm import relationship
from simplekml import Kml, Style, IconStyle, Icon, LineStyle

from tropofy.app import AppWithDataSets, Step, StepGroup
from tropofy.widgets import ExecuteFunction, SimpleGrid, KMLMap
from tropofy.database.tropofy_orm import DataSetMixin
from tropofy.database import read_write_xl

import exmod


class Intersection(DataSetMixin):
    name = Column(Text, nullable=False)
    latitude = Column(Float, nullable=False)
    longitude = Column(Float, nullable=False)

    def __init__(self, name, latitude, longitude):
        self.name = name
        self.latitude = latitude
        self.longitude = longitude

    @classmethod
    def get_table_args(cls):
        return (UniqueConstraint('data_set_id', 'name'),)


class Road(DataSetMixin):
    start_name = Column(Text, nullable=False)
    end_name = Column(Text, nullable=False)
    spanning_tree = Column(Boolean, nullable=True)

    start_intersection = relationship(Intersection, foreign_keys=[start_name])
    end_intersection = relationship(Intersection, foreign_keys=[end_name])

    def __init__(self, start_name, end_name, spanning_tree):
        self.start_name = start_name
        self.end_name = end_name
        self.spanning_tree = spanning_tree

    @classmethod
    def get_table_args(cls):
        return (
            UniqueConstraint('data_set_id', 'start_name', 'end_name'),
            ForeignKeyConstraint(['start_name', 'data_set_id'], ['intersection.name', 'intersection.data_set_id'], ondelete='CASCADE', onupdate='CASCADE'),
            ForeignKeyConstraint(['end_name', 'data_set_id'], ['intersection.name', 'intersection.data_set_id'], ondelete='CASCADE', onupdate='CASCADE')
        )
    
class IntersectionKMLMap(KMLMap):
    def get_kml(self, data_set):
        kml = Kml()

        def LongLat(l):
            return (l.longitude, l.latitude)

        mylocstyle = Style(iconstyle=IconStyle(scale=0.1, icon=Icon(href='http://maps.google.com/mapfiles/kml/paddle/blu-circle-lv.png')))
        LocsFolder = kml.newfolder(name="Intersections")
        for p in [LocsFolder.newpoint(name=str(loc.name), coords=[LongLat(loc)]) for loc in data_set.query(Intersection).all()]:
            p.style = mylocstyle

        return kml.kml()


class RoadKMLMap(KMLMap):
    def get_kml(self, data_set):
        kml = Kml()

        def LongLat(l):
            return (l.longitude, l.latitude)

        mylocstyle = Style(iconstyle=IconStyle(scale=0.1, icon=Icon(href='http://maps.google.com/mapfiles/kml/paddle/blu-circle-lv.png')))
        LocsFolder = kml.newfolder(name="Locations")
        for p in [LocsFolder.newpoint(name=str(loc.name), coords=[LongLat(loc)]) for loc in data_set.query(Intersection).all()]:
            p.style = mylocstyle

        mylinestyle = Style(linestyle=LineStyle(color='FF7171C6', width=4))
        LinesFolder = kml.newfolder(name="Lines")
        for line in [LinesFolder.newlinestring(name='line', coords=[LongLat(l.start_intersection), LongLat(l.end_intersection)]) for l in data_set.query(Road).all()]:
            line.style = mylinestyle

        return kml.kml()


class SpanningTreeKMLMap(KMLMap):
    def get_kml(self, data_set):
        kml = Kml()

        def LongLat(l):
            return (l.longitude, l.latitude)

        mylocstyle = Style(iconstyle=IconStyle(scale=0.1, icon=Icon(href='http://maps.google.com/mapfiles/kml/paddle/blu-circle-lv.png')))
        LocsFolder = kml.newfolder(name="Locations")
        for p in [LocsFolder.newpoint(name=str(loc.name), coords=[LongLat(loc)]) for loc in data_set.query(Intersection).all()]:
            p.style = mylocstyle

        mylinestyle = Style(linestyle=LineStyle(color='FF7171C6', width=4))
        LinesFolder = kml.newfolder(name="Lines")
        for line in [LinesFolder.newlinestring(name='line', coords=[LongLat(l.start_intersection), LongLat(l.end_intersection)]) for l in data_set.query(Road).filter(Road.spanning_tree == True)]:
            line.style = mylinestyle

        return kml.kml()

def haversine(lon1, lat1, lon2, lat2):
    lon1, lat1, lon2, lat2 = map(radians, [lon1, lat1, lon2, lat2])
    dlon = lon2 - lon1
    dlat = lat2 - lat1
    a = sin(dlat / 2) ** 2 + cos(lat1) * cos(lat2) * sin(dlon / 2) ** 2
    c = 2 * asin(sqrt(a))
    km = 6367 * c
    return km


class ExecuteSolverFunction(ExecuteFunction):

    execution_data_set = None

    def get_button_text(self):
        return "Find Spanning Tree"

    def execute_function(self, data_set):
        ExecuteSolverFunction.execution_data_set = data_set
        exmod.callback(self.add_road_to_spanning_tree)

        data_set.send_progress_message("Start Kruskals Algorithm")

        arcs = data_set.query(Road).all()
        nodes = data_set.query(Intersection).all()

        for road in arcs:
            start_intersection = road.start_intersection
            end_intersection = road.end_intersection
            distance = haversine(start_intersection.longitude, start_intersection.latitude, end_intersection.longitude, end_intersection.latitude)
            exmod.add_edge(distance, str(road.start_name), str(road.end_name))

        data_set.send_progress_message("Network Statistics")
        data_set.send_progress_message("Arcs: " + str(len(arcs)))
        data_set.send_progress_message("Nodes: " + str(len(nodes)))

        exmod.solve_kruskal()
        
        data_set.send_progress_message("Finish Kruskals Algorithm")


    def add_road_to_spanning_tree(self, start_name, end_name):
        road = ExecuteSolverFunction.execution_data_set.query(Road).filter_by(start_name=start_name, end_name=end_name).one()
        road.spanning_tree = True


class SpanningTreeApp(AppWithDataSets):

    def get_name(self):
        return "Minimum Spanning Tree"

    def get_examples(self):
        return {"Data for Brisbane": self.load_example_data_for_brisbane}

    def get_gui(self):
        step_group1 = StepGroup(name='Input')
        step_group1.add_step(Step(
            name='Intersections',
            widgets=[
                SimpleGrid(Intersection),
                IntersectionKMLMap()
            ]
        ))
        
        step_group1.add_step(Step(
            name='Roads',
            widgets=[
                SimpleGrid(Road),
                RoadKMLMap(),
            ]
        ))

        step_group2 = StepGroup(name='Engine')
        step_group2.add_step(Step(name='Optimise', widgets=[ExecuteSolverFunction()]))

        step_group3 = StepGroup(name='Output')
        step_group3.add_step(Step(name='Minimum Spanning Tree', widgets=[SpanningTreeKMLMap()]))

        return [step_group1, step_group2, step_group3]

    @staticmethod
    def load_example_data_for_brisbane(data_set):
        read_write_xl.load_data_from_excel_file_on_disk(data_set, data_set.app.get_path_of_file_in_app_folder('spanning_tree_example_data.xlsx'))

    def get_icon_url(self):
        return 'http://www.tropofy.com/static/css/img/tropofy_example_app_icons/spanning_tree.png'
    
    def get_home_page_content(self):
        return {'content_app_name_header': '''
                <div>
                <span style="vertical-align: middle;">Minimum Spanning Tree</span>
                <img src="http://www.tropofy.com/static/css/img/tropofy_example_app_icons/spanning_tree.png" alt="main logo" style="width:15%">
                </div>''',

                'content_single_column_app_description': '''

                <p>This app generates a minimum spanning tree, using Kruskal's algorithm, for a given road network.</p>
                <p>The input is a road network, described using intersections and their adjacent roads.  
                The output is a minimum spanning tree based on road length.  In cases where 
                the road network is not connected, the output is a forest (a set of trees).</p>
                
                <p>Sign up and give it a go!</p>
                <p>Need help or wish this app had more features? Contact us at <b>info@tropofy.com</b> to see if we can help.</p>
                ''',

                'content_double_column_app_description_1': '',
                'content_double_column_app_description_2': '',
                'content_row_2_col_1_header': '',
                'content_row_2_col_1_content': '',
                'content_row_2_col_2_header': '',
                'content_row_2_col_2_content': '',
                'content_row_2_col_3_header': '',
                'content_row_2_col_3_content': '',
                'content_row_3_col_1_header': '',
                'content_row_3_col_1_content': '',
                'content_row_3_col_2_header': '',
                'content_row_3_col_2_content': '',
                'content_row_4_col_1_header': '',

                'content_row_4_col_1_content': '''
                This app was created using the <a href="http://www.tropofy.com" target="_blank">Tropofy platform.</a>.
                '''
            }

