'''
Authors: www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

The LocalSolver this app is based on can be found at
http://www.localsolver.com/exampletour.html?file=car_sequencing.zip

Used with permission.

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import subprocess
from sqlalchemy.types import Integer, Float
from sqlalchemy.schema import Column

from tropofy.app import AppWithDataSets, Parameter, Step, StepGroup
from tropofy.widgets import ExecuteFunction, Chart, ParameterForm
from tropofy.database.tropofy_orm import DataSetMixin


BIN_WIDTH = 0.5

class SLXQueueParams(DataSetMixin):
    num_customers = Column(Integer, nullable=False)
    arrivals_per_minute = Column(Float, nullable=False)
    num_tellers = Column(Integer, nullable=False)
    mean_service_time = Column(Float, nullable=False)

class SLXTimeInQueueResults(DataSetMixin):
    lower_bound = Column(Float, nullable=False)
    percentage = Column(Float, nullable=False)

    @property
    def label(self):
        return '%s - %s' % (self.lower_bound, self.lower_bound + BIN_WIDTH)  # Todo: Parameterise Bin Width
    

class SLXTimeInSysResults(DataSetMixin):
    lower_bound = Column(Float, nullable=False)
    percentage = Column(Float, nullable=False)

    @property
    def label(self):
        return '%s - %s' % (self.lower_bound, self.lower_bound + BIN_WIDTH)  # Todo: Parameterise Bin Width


class ExecuteSLX(ExecuteFunction):
    def get_button_text(self):
        return "Solve Simple Queueing Problem with SLX"

    def execute_function(self, data_set):
        call_local_solver(data_set)


class TimeInQueueHistogram(Chart):
    def get_chart_type(self, data_set):
        return Chart.COLUMNCHART

    def get_table_schema(self, data_set):
        return {
            "time_in_queue": ("string", "Time in Queue"),
            "percentage": ("number", "%")
        }

    def get_table_data(self, data_set):
        data = []
        for row in data_set.query(SLXTimeInQueueResults).all():
            data.append({'time_in_queue': row.label, 'percentage': row.percentage/100})
        return data

    def get_column_ordering(self, data_set):
        return ["time_in_queue", "percentage"]

    def get_chart_options(self, data_set):
        return {
            'title': 'Time in Queue Summary',
            'legend': {'position': 'none'},
            'vAxis': {'format': '#%'},
            'hAxis': {
                'title': 'Time Bracket (minutes)',
            },
        }

class TimeInSysHistogram(Chart):
    def get_chart_type(self, data_set):
        return Chart.COLUMNCHART

    def get_table_schema(self, data_set):
        return {
            "time_in_system": ("string", "Time in System"),
            "percentage": ("number", "%")
        }

    def get_table_data(self, data_set):
        data = []
        for row in data_set.query(SLXTimeInSysResults).all():
            data.append({'time_in_system': row.label, 'percentage': row.percentage/100})
        return data

    def get_column_ordering(self, data_set):
        return ["time_in_system", "percentage"]

    def get_chart_options(self, data_set):
        return {
            'title': 'Time in System Summary',
            'legend': {'position': 'none'},
            'vAxis': {'format': '#%'},
            'hAxis': {
                'gridlines': {'count': 10},
                'title': 'Time Bracket (minutes)',                
            },
        }


class SLXSimpleQueueApp(AppWithDataSets):
    def get_name(self):
        return 'SLX Simple Queue'

    def get_gui(self):
        step_group1 = StepGroup(name='Enter your Data')
        step_group1.add_step(Step(
            name='Parameters',
            widgets=[ParameterForm()],
        ))
        step_group2 = StepGroup(name='Run Simulation')
        step_group2.add_step(Step(name='Simulate Queue', widgets=[ExecuteSLX()]))
        step_group3 = StepGroup(name='Results')
        step_group3.add_step(Step(name='Results', widgets=[TimeInQueueHistogram(), TimeInSysHistogram()]))

        return [step_group1, step_group2, step_group3]

    def get_examples(self):
        return {
            "Slow Line": load_data_set_slow_line,
        }

    def get_parameters(self):
        return [
            Parameter(name='num_customers', label='Number of Customers', default=100000, allowed_type=int),
            Parameter(name='arrivals_per_minute', label="Arrivals per Minute", default=3, allowed_type=float),
            Parameter(name='num_tellers', label='Number of Tellers', default=6, allowed_type=int),
            Parameter(name='mean_service_time', label='Mean Service Time', default=1.75, allowed_type=float),
        ]


    def get_home_page_content(self):
        return {
            'content_app_name_header': '''
            <div>
            <span style="vertical-align: middle;">SLX Simple Queue</span>
            <img src="http://www.tropofy.com/static/css/img/apps/facility_location.png" alt="main logo" style="width:15%">
            </div>''',

            'content_single_column_app_description': '''
            <p>A model of a bank in which incoming customers join the shortest teller queue and stick with it.</p>
            <p>This app is a proof of concept of integrating Tropofy with SLX.</p>            
            ''',

            'content_row_4_col_1_content': '''
            This app was created using the <a href="http://www.tropofy.com" target="_blank">Tropofy platform</a> and is powered by SLX.
            '''
        }


def load_data_set_slow_line(data_set):
    data_set.add(SLXQueueParams(
        num_customers=100000,
        arrivals_per_minute=3,
        num_tellers=6,
        mean_service_time=1.75,
    ))
    

def call_local_solver(data_set):
    invoke_localsolver_using_lsp_file(data_set, write_slx_input_file(data_set))

def write_slx_input_file(data_set):
    input_file_path = data_set.get_path_of_file_in_data_set_folder('input.dat')    
    f = open(input_file_path, 'w')

    f.write('%s\n' % data_set.get_param('num_customers'))
    f.write('%s\n' % data_set.get_param('arrivals_per_minute'))
    f.write('%s\n' % data_set.get_param('num_tellers'))
    f.write('%s\n' % data_set.get_param('mean_service_time'))

    f.close()
    return input_file_path


def invoke_localsolver_using_lsp_file(data_set, input_file_path):
    # Reset solution
    solution_file_path = data_set.get_path_of_file_in_data_set_folder('output.lis')
    time_in_q_histogram_source_file_path = data_set.get_path_of_file_in_data_set_folder('timeinQ.lis')
    time_in_sys_histogram_source_file_path = data_set.get_path_of_file_in_data_set_folder('timeinsys.lis')
    data_set.query(SLXTimeInQueueResults).delete()
    data_set.query(SLXTimeInSysResults).delete()

    open(solution_file_path, 'w').close()  # clear the solution files if they exist
    open(time_in_q_histogram_source_file_path, 'w').close()
    open(time_in_sys_histogram_source_file_path, 'w').close()
    
    p = subprocess.Popen(["c:\wolverine\slx\sse", "/output", solution_file_path, "quickline", input_file_path, time_in_q_histogram_source_file_path, time_in_sys_histogram_source_file_path],
        stdout=subprocess.PIPE,
        cwd=data_set.app.app_folder_path)
    out, _ = p.communicate()

    with open(solution_file_path) as f:
        content = f.readlines()

        if content:            
            data_set.send_progress_message('''<pre>%s_</pre>''' % "".join(content))
        else:
            data_set.send_progress_message(
                '''An error occured when executing SLX.'''
            )

    read_slx_histogram_data(data_set, time_in_q_histogram_source_file_path, SLXTimeInQueueResults)
    read_slx_histogram_data(data_set, time_in_sys_histogram_source_file_path, SLXTimeInSysResults)

def read_slx_histogram_data(data_set, file_path, source_class):
    try:
        with open(file_path) as f:
            content = f.readlines()

            if content:
                results = []
                for row in content:
                    result = row.split(',')
                    results.append(source_class(
                        lower_bound=result[0],
                        percentage=result[1],
                    ))
                data_set.add_all(results)
                data_set.send_progress_message('Success! \n%s' % str(results))
            else:
                raise Exception
    except:        
        data_set.send_progress_message('An error occured reading a histogram output file.')
        raise