'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from sqlalchemy.exc import IntegrityError
from sqlalchemy.orm import _mapper_registry

from tropofy.widgets import Widget
from tropofy.database.tropofy_orm import ValidationException
from tropofy.database import DBSession


class GridWidget(Widget):
    ASCENDING = 'asc'
    DESCENDING = 'desc'

    def get_type(self):
        pass

    def _get_query_column_objects(self):
        return {col.name: col for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns}

    def _get_foreign_key_column_objects(self):
        return {name: col for (name, col) in self._get_query_column_objects().items() if col.foreign_keys and col.name not in self.get_hidden_column_names()}

    def _get_foreign_key_column_names(self):
        return [k for k in self._get_foreign_key_column_objects().keys()]

    def refresh_from_db(self, data_set, request):
        server_side_processing = 'sEcho' in request.GET.keys()
        if server_side_processing:
            col_names = self.get_grids_column_names()
            ordered_col_name_sort_direction_tuples = []
            for i in xrange(int(request.GET['iSortingCols'])):
                ordered_col_name_sort_direction_tuples.append(
                    (col_names[int(request.GET["iSortCol_%s" % i])], request.GET["sSortDir_%s" % i])
                )

            objects, total_records_before_filtering, total_records_after_filtering = self.get_filtered_refresh_objects(
                data_set=data_set,
                display_start=int(request.GET['iDisplayStart']),
                display_length=int(request.GET['iDisplayLength']),
                global_search_field=request.GET['sSearch'],
                ordered_col_name_sort_direction_tuples=ordered_col_name_sort_direction_tuples,
            )
            if objects == [] or objects:  # [] means no data in table = OK to proceed. None or something else falsy means an error and shouldn't proceed.
                return {
                    'sEcho': int(request.GET['sEcho']),  # Suggested to cast to int to protect against cross-site-scripting attacks,
                    'iTotalRecords': total_records_before_filtering,
                    'iTotalDisplayRecords': total_records_after_filtering,
                    'aaData': [obj.as_json_data_row(self.get_grids_column_names()) for obj in objects]
                }
        else:
            objects = self.get_refresh_objects(data_set)
            if objects == [] or objects:  # [] means no data in table = OK to proceed. None or something else falsy means an error and shouldn't proceed.
                return {
                    'aaData': [obj.as_json_data_row(self.get_grids_column_names()) for obj in objects]
                }            

    def update_to_db(self, data, data_set_id, oper='', data_set=None):
        edited_row = []  # Get edited row from db after changed. For updating client with accurate info.
        message = ''
        success = True
        for k, v in data.iteritems():  # Convert empty string entries to None. Needed to validate against is nullable on strings. (empty strings used on non nullable elsewhere in framework. Long term fix is to change this!)
            if v == '':
                data[k] = None
        if oper == 'del':
            self.delete_row(obj_id=data['id'])
        elif oper == 'edit':
            try:
                edited_row = self.edit_row(data)
            except (ValidationException, IntegrityError) as e:
                message = e.message
                success = False
        elif oper == 'add':
            try:
                edited_row = self.add_new_row(data, data_set_id)
            except (ValidationException, IntegrityError) as e:
                message = e.message
                success = False
        else:
            return None  # TODO: Raise a 404 page not found here?
        return {'success': success, 'message': message, 'edited_row': edited_row}

    @staticmethod
    def __get_mapper_for_table(table):
        for mapper in list(_mapper_registry):
            if table in mapper._all_tables:
                return mapper
        return None

    def update_from_db(self, data, data_set_id, oper='', data_set=None):
        if oper == 'get_select_options' and self.grid_is_editable():
            column = self._get_foreign_key_column_objects()[data['column_name']]
            foreign_key_col = list(column.foreign_keys)[0].column
            query_result = None
            mapper = self.__get_mapper_for_table(foreign_key_col.table)
            if data_set_id:  # TODO: Figure out how to do this through ORM relationship if it exists!
                query_result = DBSession().query(getattr(mapper.class_, foreign_key_col.name)).filter_by(data_set_id=data_set_id).all()
            else:
                query_result = DBSession().query(getattr(mapper.class_, foreign_key_col.name)).all()
            return {'success': True, 'items': [item for row in query_result for item in row]}
        return {'success': False}

    def serialise(self):
        '''Join serialisation dict with Widget to get all params needed on client'''
        return dict(Widget.serialise(self).items() + {
            'columns': self.get_grids_column_names(),
            'hiddenCols': self.get_hidden_column_names(),
            'foreignKeyCols': self._get_foreign_key_column_names(),
            'descSortedCols': self.get_decs_sorted_cols(),  # Cols default is to sort ascending on init. If in this list they will start sorted descending.
            'nonEditabledCols': self.get_non_editable_cols(),
            'formInputTypesOfCols': self.get_column_form_input_types(),
            'gridIsEditable': self.grid_is_editable()            
        }.items())

    def grid_is_editable(self):
        raise NotImplementedError

    def get_grids_column_names(self):
        raise NotImplementedError

    def get_column_form_input_types(self):
        '''Input types to use in the form for cols. Must be in same order as get_grids_column_names. If col undefined, assume select if a foreign key, else text'''
        return []

    def get_non_editable_cols(self):
        '''Will disable input boxes for these cols on edit in the GUI.'''
        return []

    def get_decs_sorted_cols(self):
        return []

    def get_hidden_column_names(self):
        return ['id', 'data_set_id']

    def get_refresh_objects(self, data_set):
        # Todo: Put source_class on GridWidget not SimpleGrid. SimpleGrid source_class must be SQLA class. GridWidget's does not (could be Python class or collections.NamedTuple.
        # Then, this function must return a set of source_class.
        # Do this and we can derive:
        # - GridWidget.get_column_form_input_types
        # - GridWidget.get_grids_column_names (must match source_class methods). Should probably add option to put labels on these so actual col names in grid change
        # - GridWidget.get_col_python_type (Doesn't exist yet - probably exists on FormWidget to determine saving add/edit string->python type conversion)

        # What to do about foreign keys? GridWidget shouldn't care! All on FormWidget. FormWidget select type get options & save options.

        # How to connect GridWidget and FormWidget? FormWidget is embedded class.
        #    - Listeners: 
        #        - (Python) FormWidget.on_load_listeners = "specific grid widget somehow"
        #        - (Python) FormWidget.on_submit_success_listeners = "specific grid widget somehow"
        #        - (JS) FormWidget: for widgets in onLoadListeners: widget.onFormLoad(formvm, data, edit/add) {populate form with grid data}
        #        - (JS) FormWidget: for widgets in onSubmitSuccesslisteners: widget.onSubmitSuccess(formvm, data, edit/add) {add/edit row in grid}
        raise NotImplementedError

    def get_filtered_refresh_objects(self, data_set, display_start, display_length, global_search_field, ordered_col_name_sort_direction_tuples):
        """
        :returns: (list of filtered objects to display in grid for this page/search, total records in data set before filter, total records in data set after filter)
        :rtype: tuple - (<object list>, <int>, <int>)
        """
        raise NotImplementedError        

    def edit_row(self, data):
        raise NotImplementedError

    def add_new_row(self, data, data_set_id):
        raise NotImplementedError

    def delete_row(self, obj_id):
        raise NotImplementedError

    def get_table_hierachy_from_inheritance_chain(self):  # This is confusing. Shouldn't live on GridWidget - should be an implementation detail on a subclass.
        raise NotImplementedError
