'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from tropofy.widgets.widgets import Widget


class StaticImage(Widget):
    """A static image displayed to screen.

    .. note:: To use this in an app, you must install the Python library `pillow <https://pypi.python.org/pypi/Pillow/2.1.0>`_.
       It requires compilation of C libraries.

       .. code-block:: bash

          (tropofy_env) $ pip install pillow
    
    More detailed installation instructions can be found at :ref:`Compiling C++ Python Extensions<compiling_c_extensions>`.

    Images may come from external urls, or reference local image files generated with an app.
    Use this widget to display images create with libraries such as `Matplotlib <http://matplotlib.org/>`_,
    and `R <http://www.r-project.org/>`_. The example below is a common usage pattern of StaticImage:

    1. Create and save an image within your app:

       .. code-block:: python

          #img is an in memory PLT.Image object created by your app
          data_set.save_image(name="output.png", image=img)

    2. Implement a StaticImage widget as follow:

       .. code-block:: python

          class OutputPlot(StaticImage):
              def get_file_path(self, data_set):
                  return data_set.get_image_path('output.png')

    3. Include your StaticImage widget within :func:`tropofy.app.AppWithDataSets.get_gui`


    """

    def get_type(self):
        return "ImageWidget"

    def refresh_from_db(self, data_set, request):
        file_path = self.get_file_path(data_set)
        return {
            'get_file_path': file_path if file_path else "http://www.tropofy.com/static/css/img/error.png",
        }

    def get_file_path(self, data_set):
        """A path to a static image file.

        The path may be a url or file on disk. For a file on disk generated by your app, save the file in your app using
        :func:`tropofy.app.AppDataSet.save_image` and use :func:`tropofy.app.AppDataSet.get_image_path` to get the
        path of a file to serve statically.

        :param data_set: Advanced use only. For programatically defining file path, from the current DataSet.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: string

        For example, to point to a hosted file at a url:

        .. code-block:: python

           def get_file_path(data_set):
               return 'http://www.tropofy.com/static/css/img/tropofy_example_app_icons/travelling_salesman.png'

        To return the file path of an image saved with :func:`tropofy.app.AppDataSet.save_image`:

        .. code-block:: python

           class OutputPlot(StaticImage):
               def get_file_path(self, data_set):
                   return data_set.get_image_path('output.png')

        """
        raise NotImplementedError
