'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import requests
import json
from tropofy.database.tropofy_orm import ExtendedJsonEncoder
import pkg_resources
import pyramid.httpexceptions as exc
from pyramid.response import Response
from pyramid import threadlocal
from pyramid.security import authenticated_userid, remember, forget
from pyramid.view import view_config, forbidden_view_config, notfound_view_config


def route_url_to_gui_server():
    request = threadlocal.get_current_request()
    return request.registry.settings.get('custom.gui_server_url') + request.path


def proxy_request(data, url=None):
    return post_to_gui_server(url=url if url else route_url_to_gui_server(), data=data)


def post_to_gui_server(url, data):
    request = threadlocal.get_current_request()
    key = request.app.get_published_license_key() if request.app and request.app.get_published_license_key() else request.registry.settings.get('custom.license_key')
    data.update({
        'license_key': key,
        'authenticated_user': authenticated_userid(request),
        'client_ip': request.client_addr,
        'tropofy_version': pkg_resources.get_distribution("tropofy").version,
    })
    
    response = requests.post(url, data=json.dumps(data, cls=ExtendedJsonEncoder), headers={'Content-type': 'application/json', 'Accept': 'text/plain'})
    check_response_for_success(response)
    return response


def log_activity(request, description):
    url = request.registry.settings.get('custom.gui_server_url') + '/' + request.app.url_name + '/log_activity'
    proxy_request(
        data={'description': description, 'from_compute_node': True},
        url=url
    )


@view_config(route_name='run_app', permission='run_app')
def run_app(request):
    app = request.app
    emulator_dict = {
        'get_app_name': app.name,
        'get_step_groups': [step_group.serialise() for step_group in app.get_ordered_step_groups()],
        'get_app_icon_url': app.get_icon_url(),
        'disable_forced_linear_workflow': app.disable_forced_linear_workflow(),
    }
    result = proxy_request({'emulator_dict': emulator_dict})
    return Response(result.content)


@view_config(route_name='redirect_to_run_app')
def redirect_to_run_app(request):
    if request.only_app:
        return exc.HTTPFound(
            location=request.route_url('run_app', app_url_name=request.only_app.url_name)
        )
    else:
        raise exc.HTTPNotFound()


@view_config(route_name='update_widget_to_db', renderer='json', permission='run_app')
def update_widget_to_db(request):
    response = request.widget.update_to_db(data=dict(request.POST), data_set_id=request.data_set_id, oper=request.matchdict['oper'], data_set=request.data_set)
    if request.matchdict['oper'] == 'needs_text_response':
        request.response.content_type = 'text/plain'
    return response


@view_config(route_name='update_widget_from_db', renderer='json', permission='run_app')
def update_widget_from_db(request):
    return request.widget.update_from_db(data=dict(request.POST), data_set_id=request.data_set_id, oper=request.matchdict['oper'], data_set=request.data_set)


@view_config(route_name='refresh_widget_from_db', renderer='json')  # , xhr=True, renderer='json', permission='run_app')
def refresh_widget_from_db(request):
    widget_parameters = request.widget.refresh_from_db(data_set=request.data_set, request=request)
    if request.widget.get_type() is "KMLMapWidget":  # too much to send up and back
        response = Response(widget_parameters['get_kml'])
        response.content_type = 'text/xml'  # Must explicitly set for ie
        return response
    elif request.widget.get_type() is "CustomExportWidget":  # too much to send up and back
        response = Response(widget_parameters['get_data'])
        response.content_type = 'text/plain'  # Must explicitly set for ie
        return response
    elif request.widget.get_type() in ["FormWidget", "GridWidget", "SimpleGrid"]:  # Don't need to send up
        return widget_parameters
    else:
        data = {'widget_type': request.widget.get_type(), 'widget_parameters': widget_parameters}
        response = proxy_request(data)
        response = json.loads(response.text) # Todo: Change to response.content ? 

        if request.widget.get_type() == "KMLMapWidget":  # On client, Openlayers expects the KML. No chance to decode from json.
            return Response(response['kml'])
        return response


@view_config(route_name='export_data_set', permission='run_app')
def export_data_set(request):
    if request.data_set_id:
        log_activity(request, ("Export data set: '{name}'".format(name=request.data_set.name)))
    else:
        log_activity(request, 'Export Excel Template')
    return request.app.export_data_set(request.data_set)


### Data Set Management ###
@view_config(route_name='create_new_data_set', xhr=True, renderer='json', permission='run_app')
def create_new_data_set(request):
    if request.app is not None:
        data_set = request.app.create_new_data_set_for_user(
            authenticated_userid(request),
            request.matchdict['data_set_template'],
            request.matchdict['data_set_name']
        )
        return {
            'success': True,
            'dataSetId': data_set.id
        }
    return {'success': False}


@view_config(route_name='start_app_with_data_set', xhr=True, renderer='json', permission='run_app')
def start_app_with_data_set(request):
    data_set = request.data_set
    data_set.update_time_accessed()
    return {
        'currentStepIndex': data_set.current_step_index,
        'latestEnabledStepIndex': data_set.latest_enabled_step_index,
        'dataSetId': data_set.id,
        'dataSetName': data_set.name,
    }


@view_config(route_name='set_data_set_current_step', xhr=True, renderer='json', permission='run_app')
def set_data_set_current_step(request):
    current_step_index = int(request.matchdict['current_step_index'])
    if request.data_set:  # some apps have no data set
        if current_step_index > 0:  # Don't store if current step is data_set_selection step.
            request.data_set.current_step_index = current_step_index
            if request.data_set.latest_enabled_step_index < current_step_index:
                request.data_set.latest_enabled_step_index = current_step_index
    return {'success': True}


@view_config(route_name='login', renderer='json', request_method='POST')
def login(request):
    data = {
        'login': request.POST.get('login'),
        'password': request.POST.get('password'),
    }

    if proxy_request(data).status_code == 200:
        headers = remember(request, request.POST['login'])
        request.response.headerlist.extend(headers)
        return {'success': True}


@view_config(route_name='logout', renderer='string')
def logout(request):
    headers = forget(request)

    mako_render_dict = request.app.get_home_page_configuration()
    data = {'mako_render_dict': mako_render_dict}
    url = request.registry.settings.get('custom.gui_server_url') + '/' + request.app.url_name + '/homepage'
    result = proxy_request(data, url)
    return Response(result.content, headers=headers)


@forbidden_view_config(renderer='string')
def forbidden(request):
    return exc.HTTPFound(
        location=request.route_url('logout', app_url_name=request.matchdict.get('app_url_name'))
    )


@notfound_view_config()
def notfound(exc, request):
    """Catches all 'raise HTTPNotFound'.

    Todo: Plain text reponse for now. Should render to a template. Template should be on GUI server though so need to proxy.
    """
    response_str = "Tropofy 404: We're sorry, but the page you are looking for could not be found."
    if request.matchdict and request.matchdict.get('app_url_name'):
        response_str += " You should check the spelling of the app you are trying to access: " + \
            "'" + request.matchdict['app_url_name'] + "'"
    response = Response(response_str)
    response.status = 404
    return response


def check_response_for_success(response):
    # (1) HTTPForbidden thrown on GUI server
    if response.status_code == 403:
        raise exc.HTTPForbidden()

    # (2) HTTPNotFound thrown on GUI server - e.g. if app_name is not valid
    if response.status_code == 404:
        raise exc.HTTPNotFound()

    # (3) NonCompatibleComputeNodeException thrown on GUI server.
    if response.status_code == 409:
        pass  # Let response get rendered.

    if response.status_code == 500:
        raise Exception(response.text)

    # (2) Tropofy caught failure
    success = True
    try:
        response = json.loads(response.text)
        success = response.get('success')
        message = response.get('message')
    except:
        pass
    if not success and success is not None:
        raise Exception(message)


@view_config(route_name='favicon')
def favicon(response):
    return Response()


@view_config(route_name='healthcheck')
def healthcheck(response):
    return Response()