'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import json
import inspect
import collections
from datetime import datetime, date, time
from sqlalchemy.schema import Column, ForeignKey, ForeignKeyConstraint
from sqlalchemy import event
from sqlalchemy.types import Integer, Text, Unicode, Float, DateTime, Date, Time, Boolean
from sqlalchemy.ext.declarative import declarative_base, declared_attr, DeclarativeMeta

from pyramid.renderers import JSON


class DynamicSchemaName(object):
    schema_names_in_use = []
    use_app_schemas = None
    schema_name = None
    COMPUTE_NODE_SCHEMA_NAME = 'tropofy_compute'
    schema_name_has_been_used = False  # Track for assertion. Need to ensure when use_app_schemas is set, that get_schema_qualified_table_name_if_using_schemas hasn't been called.

    @classmethod  # Would be good to be a @property, but apparently can't do @property setters on classmethods.
    def set_schema_name(cls, value):
        """
        :type value: str
        """
        if not cls.use_app_schemas:
            cls.schema_name = None
        else:
            if value and value in cls.schema_names_in_use:
                raise Exception("Schema exception: Multiple apps attempting to be created with the same schema name '%s'. Schema name for an app is automatically generated from the name of the app .py file containing AppWithDataSets derived class." % value)
            cls.schema_name = value
            cls.schema_names_in_use.append(value)

    @classmethod  # Would be good to be a @property, but apparently can't do @property setters on classmethods.
    def set_use_app_schemas(cls, value):
        """
        :type value: bool
        """
        assert not cls.schema_name_has_been_used, 'DynamicSchemaName.use_app_schemas accessed before assignmed. DynamicSchemaName.set_use_app_schemas must be called before any table creation to ensure schemas are used correctly. This is why some imports (Especially AppDataSet) are called within main.'
        cls.use_app_schemas = value


    @classmethod
    def get_schema_qualified_table_name_if_using_schemas(cls, table_name):
        cls.schema_name_has_been_used = True
        return  '%s%s' % (
            DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME + '.' if DynamicSchemaName.use_app_schemas else '',
            table_name
        )
    

class ORMAncestor(object):
    id = Column(Integer, primary_key=True)

    @declared_attr
    def __tablename__(cls):
        return cls.__name__.lower()

    def _base_blacklist(self):
        '''Blacklist list of which properties not to include in JSON'''
        return ['_sa_instance_state']

    def _json_blacklist(self):
        '''Blacklist to overide'''
        return []

    def __json__(self, request, fields_to_expand=[]):  # http://stackoverflow.com/questions/5022066/how-to-serialize-sqlalchemy-result-to-json & https://gist.github.com/pjenvey/3808830
        '''Converts object to json ready dict. Need to add encoders for non Json ready types (such as Date).'''
        blacklist = set(self._base_blacklist())
        blacklist.update(self._json_blacklist())

        dict_repr = {}
        for k, v in self.__dict__.items():
            if k not in blacklist:
                try:
                    # 2) Need to use only functions by name otherwise they return as "<function get_kml_for_downloading at 0x0000000005364048>" where the mem address changes
                    if isinstance(v, DeclarativeMeta):
                        v = str(v)
                    elif inspect.isroutine(v):
                        v = v.func_name
                    json.dumps(v, cls=ExtendedJsonEncoder)  # this will fail on non-encodable values, like other classes
                    dict_repr[k] = v
                except TypeError:
                    dict_repr[k] = None

        # a json-encodable dict
        return dict_repr

    def as_json_data_row(self, ordered_field_names):
        '''Returns json safe row of field vals in order of ordered_field_names.'''
        fields = self.__json__(None)
        row = []
        for field in ordered_field_names:
            value = fields.get(field, None)  # Will be None if can't match col or value in db is None.
            row.append(value if value is not None else '')
        return row

from abc import ABCMeta
class AllowABCsToBeInheritedFromInMultipleInheritanceCases(ABCMeta, DeclarativeMeta):
    pass

ORMBase = declarative_base(cls=ORMAncestor, metaclass=AllowABCsToBeInheritedFromInMultipleInheritanceCases)

class TableConstructionException(Exception):
    pass

class TropofyDbMetaData(object):
    metadata = ORMBase.metadata

    TableNameAndSchema = collections.namedtuple('TableNameAndSchema', ['name', 'schema'])

    @classmethod
    def get_dependency_sorted_tables(cls, table_name_and_schemas):
        """
        Returns list of tables that match a given table name and schema.

        :type table_name_and_schemas: list of TropofyDbMetaData.TableNameAndSchema

        :returns: list of sqlalchemy.sql.schema.Table
        """
        return [table for table in cls.metadata.sorted_tables if (table.name, table.schema) in table_name_and_schemas]


class DataSetMixin(ORMBase):
    """Used as mixin for all developer defined models."""
    __abstract__ = True

    @declared_attr
    def data_set_id(cls):
        return Column(Integer, ForeignKey(DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas('appdataset.id'), ondelete='CASCADE'))

    @declared_attr
    def __table_args__(cls):
        __table_args__ = cls.get_table_args()

        if DynamicSchemaName.use_app_schemas:
            __table_args__ += ({'schema': DynamicSchemaName.schema_name},)

            # Add schema to ForeignKeyConstraint's
            for arg in __table_args__:
                if type(arg) is ForeignKeyConstraint:
                    for col, fk in arg._elements.items():
                        arg._elements[col] = ForeignKey(
                            DynamicSchemaName.schema_name + '.' + fk._colspec,
                            _constraint=arg,
                            name=arg.name,
                            onupdate=arg.onupdate,
                            ondelete=arg.ondelete,
                            use_alter=arg.use_alter,
                            link_to_name=arg.link_to_name,
                            match=arg.match
                        )
        return __table_args__

    @classmethod
    def get_table_args(cls):
        return ()


'''def datetime_adapter(obj, request):
    return obj.strftime('%Y-%m-%d - %I:%M %p')  # Todo: Format client side. Return obj.isoformat here (seems standard)

def time_adapter(obj, request):
    return obj.strftime('%I:%M %p')

def date_adapter(obj, request):
    return obj.strftime('%Y-%m-%d')  # Todo: Format client side. Return obj.isoformat here (seems standard)
'''

class ExtendedJsonEncoder(json.JSONEncoder):
    """Custom JSON encoding. Matching decoding...

    By overiding 'default', we can provide custom JSON encoding (date, times, etc.). However, Pyramid also does this,
    (e.g. to enable ORMAncestor.__json__) and will actually overide our ExtendedJsonEncoder.default function. It does
    this, by providing a value for 'default' to json.JSONEncoder.__init__().

    Below we overide json.JSONEncoder.__init__() to intercept the default param to ensure our default is called. The
    value of default is stored in self.second_default so that it is called after ExtendedJsonEncoder.default.
    """  # Todo: Specify where decoding is!
    def __init__(self, skipkeys=False, ensure_ascii=True,
            check_circular=True, allow_nan=True, sort_keys=False,
            indent=None, separators=None, encoding='utf-8', default=None):

        self.second_default = default

        super(ExtendedJsonEncoder, self).__init__(skipkeys=skipkeys, ensure_ascii=ensure_ascii,
            check_circular=check_circular, allow_nan=allow_nan, sort_keys=sort_keys,
            indent=indent, separators=separators, encoding=encoding, default=None)

    def default(self, obj):
        if isinstance(obj, time):
            return obj.strftime('%H:%M:%S')
        elif isinstance(obj, datetime):  # Must come before 'date' as isinstance(datetime_obj, date) == True
            return obj.strftime('%Y-%m-%d - %H:%M:%S')            
        elif isinstance(obj, date):
            return obj.strftime('%Y-%m-%d')
        
        else:
            if self.second_default:
                return self.second_default(obj)
            else:
                return super(ExtendedJsonEncoder, self).default(obj)


# Custom json_renderer for Pyramid
json_renderer = JSON(cls=ExtendedJsonEncoder)


# Validation
class ValidationException(Exception):
    """"""
    def __init__(self, name=None, value=None, message=None):
        self.message = "Validation error"
        if name is not None:
            self.message += ": %s" % name
            if value is not None:
                self.message += " = '%s'" % str(value)
        self.message += '.'
        if message is not None:
            self.message += ' %s' % message

    def __str__(self):
        return self.message


class Validator():
    @classmethod
    def validate(cls, value, type, name):
        '''Validates that 'value' is of type 'type'. Converts if able. 'name' helps give meaningful error msg's.'''        
        validator = cls._get_validator(type)
        if value is not None and validator:
            return validator(value, name)
        else:
            return value

    @classmethod
    def _get_validator(cls, validating_type):
        if type(validating_type) is type:
            validating_type = validating_type.__name__

        validators = {
            Integer: Validator._validate_int,
            Float: Validator._validate_float,
            Text: Validator._validate_string,
            Unicode: Validator._validate_string,
            Date: Validator._validate_date,
            Time: Validator._validate_time,
            DateTime: Validator._validate_datetime,
            Boolean: Validator._validate_boolean,
            'int': Validator._validate_int,
            'float': Validator._validate_float,
            'str': Validator._validate_string,
            'date': Validator._validate_date,
            'time': Validator._validate_time,
            'datetime': Validator._validate_datetime,
            'bool': Validator._validate_boolean,
        }
        return validators.get(validating_type)

    @classmethod
    def _validate_int(cls, value, name):
        if isinstance(value, basestring):
            value = value.replace(',', '')
        if not isinstance(value, int):
            try:
                value = int(value)
            except:
                pass
        if not isinstance(value, int):
            raise ValidationException(name=name, value=value, message='Expected integer')
        return value

    @classmethod
    def _validate_float(cls, value, name):
        if isinstance(value, basestring):
            value = value.replace(',', '')
        if not isinstance(value, float):
            try:
                value = float(value)
            except:
                pass
        if not isinstance(value, float):
            raise ValidationException(name=name, value=value, message='Expected decimal')
        return value

    @classmethod
    def _validate_string(cls, value, name):
        """
        Validates is str or unicode. SQLAlchemy 'Text' can store either. Should be using unicode wherever possible as this allow for non ascii chars.
        Converting to str(value) will try and convert to ascii (very limited char set). In Python 3, 'str' is actually 'unicode' and old 'str' functionality is 'bytes'
        """
        try:
            if type(value) is not unicode:
                value = str(value)
        except:
            raise ValidationException(name=name, value=value, message='Expected string')
        return value

    @classmethod
    def _validate_boolean(cls, value, name):
        """Boolean is subclass of int in Python. False == 0, True == 1"""
        try:  # Will convert "false" and "true" to bool. Help with processing bools from js input.
            value = json.loads(value)
        except:
            pass
        try:
            value = bool(value)
        except:
            raise ValidationException(name=name, value=value, message='Expected boolean')
        return value

    @classmethod
    def _validate_time(cls, value, name):
        if isinstance(value, basestring):
            try:
                value = datetime.strptime(value, '%H:%M:%S').time()
            except:
                pass
        if not isinstance(value, time):        
            raise ValidationException(name=name, value=value, message='Expected time')
        return value

    @classmethod
    def _validate_date(cls, value, name):
        if isinstance(value, basestring):
            try:
                value = datetime.strptime(value, '%Y-%m-%d').date()
            except:
                pass
        if not isinstance(value, date):
            raise ValidationException(name=name, value=value, message='Expected date')
        return value

    @classmethod
    def _validate_datetime(cls, value, name):
        if not isinstance(value, datetime):
            raise ValidationException(name=name, value=value, message='Expected datetime')
        return value


# this event is called whenever an attribute
# on a class is instrumented
@event.listens_for(ORMAncestor, 'attribute_instrument')
def configure_listener(class_, key, inst):
    if not hasattr(inst.property, 'columns'):
        return
    # this event is called whenever a "set"
    # occurs on that instrumented attribute

    @event.listens_for(inst, "set", retval=True)
    def set_(instance, value, oldvalue, initiator):
        col = inst.property.columns[0]
        if value == '' and col.type.python_type in [int, float]:
            value = None
        if value is None and col.default:  # If a default exists, set that to value if the value is None.
            value = col.default.arg
        if value is None and not col.nullable:
            py_type = col.type.python_type
            expected_type_str = 'decimal' if py_type is float else py_type.__name__
            raise ValidationException(name=col.name, value=value, message='Expected %s' % expected_type_str)
        return Validator.validate(value, col.type.__class__, col.name)


# Enable foreign_keys in sqlite
from sqlalchemy.engine import Engine
from sqlalchemy import event
import sqlite3


@event.listens_for(Engine, "connect")
def set_sqlite_pragma(dbapi_connection, connection_record):
    if type(dbapi_connection) is sqlite3.Connection:
        cursor = dbapi_connection.cursor()
        cursor.execute("PRAGMA foreign_keys=ON")
        cursor.close()
