"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and governed by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

from sqlalchemy.types import Text, Float
from sqlalchemy.schema import Column, ForeignKeyConstraint, UniqueConstraint
from sqlalchemy.orm import relationship
from simplekml import Kml, Style, IconStyle, Icon, LineStyle

from tropofy.app import AppWithDataSets, Step, StepGroup
from tropofy.widgets import SimpleGrid, KMLMap
from tropofy.database.tropofy_orm import DataSetMixin


class Location(DataSetMixin):
    name = Column(Text, nullable=False)
    latitude = Column(Float, nullable=False)
    longitude = Column(Float, nullable=False)

    def __init__(self, name, latitude, longitude):
        self.name = name
        self.latitude = latitude
        self.longitude = longitude

    @classmethod
    def get_table_args(cls):
        return (UniqueConstraint('data_set_id', 'name', name='_location_uc'),)


class Line(DataSetMixin):
    start_location_name = Column(Text, nullable=False)
    end_location_name = Column(Text, nullable=False)

    # The primaryjoin argument to relationship is only needed when there is ambiguity
    start_location = relationship(Location, primaryjoin="and_(Line.data_set_id==Location.data_set_id, Line.start_location_name==Location.name)")
    end_location = relationship(Location, primaryjoin="and_(Line.data_set_id==Location.data_set_id, Line.end_location_name==Location.name)")

    def __init__(self, start_location_name, end_location_name):
        self.start_location_name = start_location_name
        self.end_location_name = end_location_name

    @classmethod
    def get_table_args(cls):
        return (
            ForeignKeyConstraint(['start_location_name', 'data_set_id'], ['location.name', 'location.data_set_id'], ondelete='CASCADE', onupdate='CASCADE'),
            ForeignKeyConstraint(['end_location_name', 'data_set_id'], ['location.name', 'location.data_set_id'], ondelete='CASCADE', onupdate='CASCADE')
        )


class MyKMLMap(KMLMap):
    def get_kml(self, data_set):

        kml = Kml()

        def LongLat(l):
            return (l.longitude, l.latitude)

        mylocstyle = Style(iconstyle=IconStyle(scale=0.8, icon=Icon(href='http://maps.google.com/mapfiles/kml/paddle/blu-circle-lv.png')))
        LocsFolder = kml.newfolder(name="Locations")
        for p in [LocsFolder.newpoint(name=loc.name, coords=[LongLat(loc)]) for loc in data_set.query(Location).all()]:
            p.style = mylocstyle

        mylinestyle = Style(linestyle=LineStyle(color='FF00F5FF', width=4))
        LinesFolder = kml.newfolder(name="Lines")
        for line in [LinesFolder.newlinestring(name='line', coords=[LongLat(l.start_location), LongLat(l.end_location)]) for l in data_set.query(Line).all()]:
            line.style = mylinestyle

        return kml.kml()


class MyKMLGeneratorApp(AppWithDataSets):

    def get_name(self):
        return "KML Generation"

    def get_examples(self):
        return {"Demo data for Brisbane": self.load_example_data_for_brisbane}

    def get_gui(self):
        step_group1 = StepGroup(name='Enter your data')
        step_group1.add_step(Step(name='Enter your locations', widgets=[SimpleGrid(Location)]))
        step_group1.add_step(Step(name='Enter your lines', widgets=[SimpleGrid(Line)]))

        step_group2 = StepGroup(name='View KML')
        step_group2.add_step(Step(name='View your KML', widgets=[MyKMLMap()]))

        return [step_group1, step_group2]

    @staticmethod
    def load_example_data_for_brisbane(data_set):
        locs = []
        locs.append(Location("CLAYFIELD", -27.417536, 153.056677))
        locs.append(Location("SANDGATE", -27.321538, 153.069267))
        locs.append(Location("KIPPA-RING", -27.226494, 153.085287))
        locs.append(Location("CHERMSIDE WEST", -27.37862, 153.018257))
        locs.append(Location("EVERTON PARK", -27.406931, 152.992265))
        locs.append(Location("MILTON", -27.471324, 153.004781))
        locs.append(Location("KENMORE HILLS", -27.49615, 152.926702))
        locs.append(Location("WACOL", -27.590778, 152.929768))
        locs.append(Location("ARCHERFIELD", -27.568388, 153.024165))
        data_set.add_all(locs)
        MyKMLGeneratorApp.load_example_lines(locs, data_set)

    @staticmethod
    def load_example_lines(locations, data_set):
        lines = []
        lines.append(Line(locations[0].name, locations[1].name))
        lines.append(Line(locations[1].name, locations[2].name))
        lines.append(Line(locations[2].name, locations[3].name))
        lines.append(Line(locations[3].name, locations[4].name))
        lines.append(Line(locations[4].name, locations[5].name))
        lines.append(Line(locations[5].name, locations[6].name))
        lines.append(Line(locations[6].name, locations[7].name))
        lines.append(Line(locations[7].name, locations[8].name))
        data_set.add_all(lines)

    def get_icon_url(self):
        return 'http://www.tropofy.com/static/css/img/tropofy_example_app_icons/kml_generation.png'
    
    def get_home_page_content(self):
        return {'content_app_name_header': '''
                <div>
                <span style="vertical-align: middle;">KML Generation</span>
                <img src="http://www.tropofy.com/static/css/img/tropofy_example_app_icons/kml_generation.png" alt="main logo" style="width:15%">
                </div>''',

                'content_single_column_app_description': '''

                <p>Ever wanted to simply view a bunch of geocoded locations and/or connections on a map online, and then download 
                the map as a <a href="https://www.google.com/earth/index.html" target="_blank">KML</a> file to share with others.<p>
                <p>Then this app is exactly what you are looking for! </p>
                <p>Sign up and give it a go!</p>
                <p>Need help or wish this app had more features, contact us at <b>info@tropofy.com</b> to see if we can help</p>
                ''',

                'content_double_column_app_description_1': '',
                'content_double_column_app_description_2': '',
                'content_row_2_col_1_header': '',
                'content_row_2_col_1_content': '',
                'content_row_2_col_2_header': '',
                'content_row_2_col_2_content': '',
                'content_row_2_col_3_header': '',
                'content_row_2_col_3_content': '',
                'content_row_3_col_1_header': '',
                'content_row_3_col_1_content': '',
                'content_row_3_col_2_header': '',
                'content_row_3_col_2_content': '',
                'content_row_4_col_1_header': '',

                'content_row_4_col_1_content': '''
                This app was created using the <a href="http://www.tropofy.com" target="_blank">Tropofy platform</a>.
                '''
            }
