'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from datetime import datetime as dt
from datetime import timedelta
import time
import os
import shutil
import json
import transaction
from datetime import datetime
from pyramid import threadlocal
from sqlalchemy.types import Text, Integer, DateTime, Boolean
from sqlalchemy.schema import Column, UniqueConstraint
from sqlalchemy.orm import relationship
from sqlalchemy import event
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.exc import OperationalError
from tropofy.views import post_to_gui_server
from tropofy.database.tropofy_orm import ORMBase, DataSetMixin, ExtendedJsonEncoder, Validator, ValidationException
from tropofy.database import DBSession
from tropofy.app import LoadedApps
from tropofy.database.tropofy_orm import DynamicSchemaName


class DataSetDBInterface(object):

    def __init__(self, data_set):
        super(DataSetDBInterface, self).__init__()
        self.data_set = data_set

    def add(self, obj):
        obj.data_set_id = self.data_set.id
        DBSession().add(obj)
        return obj

    def flush(self):
        DBSession().flush()

    def add_all(self, objs):
        for obj in objs:
            obj.data_set_id = self.data_set.id
        DBSession().add_all(objs)

    def query(self, *args):
        return DBSession().query(*args).filter_by(data_set_id=self.data_set.id)


class AppDataSet(ORMBase):
    """A Data Set represents a collection of information relating to an App and owned by a User, a User
    could have many Data Sets all relating to one App.

    The :class:`AppDataSet` provides an interface for interacting with the Data Set.  The example below
    shows how a :class:`AppDataSet` is used,

    .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_5.py
       :pyobject: load_example_data

    In addition, :class:`AppDataSet` exposes the SQLAlchemy :class:`Query` class.  This provides a
    mechanism for easily searching a Data Set.  The example below shows how a Data Set can be queried,

    .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
       :pyobject: MyKMLMap.get_kml

    """

    type_str = Column('type', Text)
    app_name = Column(Text, nullable=False)
    user_email = Column(Text, nullable=False)
    name = Column(Text)
    created_datetime = Column(DateTime)
    last_accessed_datetime = Column(DateTime)
    current_step_index = Column(Integer, nullable=False)
    latest_enabled_step_index = Column(Integer, nullable=False)
    execute_functions_info = Column(Text)
    saved_image_references = relationship('SavedImageReference')
    
    __table_args__ = (
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {},
    )

    def __init__(self, app, user_email, name=''):
        self.app_name = app.name
        self.user_email = user_email
        self.name = name if name != '' else 'New data set'
        self.created_datetime = dt.now()
        self.current_step_index = 1  # Default step to show after data_set selected. 1 not 0 as data_set_selection is step 0.
        self.latest_enabled_step_index = 1
        self.execute_functions_info = ''

    def send_progress_message(self, message):
        """Sends messages to the user about the progress of some operation.

        :param message: Adds ``message`` to the current progress message for the data set.
        :type message: string

        """
         # A unit test might be running meaning there is no threadlocal.get_current_request()
        try:
            # self.execute_function_info.update_msgs_wo_orm(msg=message, engine=self.app.engine)  # (Slows things down a lot... sending progress msg to screen through writing to DB isn't great...)
            self.execute_function_info.update_msgs(msg=message)
            print(message)
        except:
            pass

    def init_function_execution_info(self, execute_function_widget):
        self.add(FunctionExecutionInfo(execute_function_widget.id))

    @property
    def execute_function_info(self):
        widget_id = threadlocal.get_current_request().widget.id
        try:
            return self.query(FunctionExecutionInfo).filter_by(widget_id=widget_id).one()
        except NoResultFound:
            new_fn_exec_info = FunctionExecutionInfo(widget_id)
            self.add(new_fn_exec_info)
            return new_fn_exec_info

    @property
    def created_datetime_str(self):
        return self.created_datetime.strftime('%Y-%m-%d - %H:%M%p') if self.created_datetime else ''

    @property
    def last_accessed_datetime_str(self):
        return self.last_accessed_datetime.strftime('%Y-%m-%d - %H:%M%p') if self.last_accessed_datetime else ''

    @property
    def app_url_name(self):  # Not added as column because not backward compatible with version 0.29.1
        for app in LoadedApps.apps.values():
            if app.name == self.app_name:
                return app.url_name

    def update_time_accessed(self):
        self.last_accessed_datetime = dt.now()

    @property
    def app(self):
        return LoadedApps.apps.get(self.app_url_name)

    @property
    def file_save_folder(self):
        """Returns a path on disk of a folder specific to the data set to save files to.

        Additionally, creates the folder for the data set if it does not yet exist.
        """
        folder = os.path.join(self.app.app_folder_path, 'data_set_data', 'data_set_' + str(self.id))
        if not os.path.exists(folder):
            os.makedirs(folder)
        return folder

    def add(self, obj):
        """Adds a TropofyBase object to the Data Set.

        :param obj: Adds ``obj`` to the Data Set.
        :type obj: :class:`tropofy.app.TropofyBase()`

        """
        session = DataSetDBInterface(self)
        session.add(obj)
        session.flush()
        return obj

    def add_all(self, objects):
        """Adds a list of TropofyBase objects to the Data Set.  The example below
        shows how :func:`add_all` is used,

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_5.py
           :pyobject: load_example_data

        :param objs: Adds ``obj`` to the Data Set.
        :type objs: list

        """
        session = DataSetDBInterface(self)
        session.add_all(objects)
        session.flush()


    def query(self, *args):
        """Returns an SQLAlchemy :class:`Query` object for the Data Set.  The example below
        shows how :func:`query` is used,

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
           :pyobject: MyKMLMap.get_kml

        :param args: Classes to query, replicates SQLA functionality.
        :type args: Variable number of arguments. Each should be of type :class:`tropofy.app.TropofyBase()` or a Column on such a class. See the `SQLA query docs <http://docs.sqlalchemy.org/en/rel_0_9/orm/query.html#the-query-object>`_ for help.

        :returns: Returns a ``Query`` object populated by with all objects of type ``source_class``
        """
        session = DataSetDBInterface(self)
        return session.query(*args).filter()

    def save_image(self, name, image):
        """Saves a file to disk, and manages its lifetime and location..

        :param name: Name of file. Must include file type extension such at .png or .jpg.
        :type name: string
        :param file: File stream
        :type file: Image
        :rtype: None
        """

        img_reference = self._get_image_reference_by_name(name)
        if img_reference:
            img_reference.save_image(image)
        else:
            img_reference = SavedImageReference(image_name=name)
            self.add(img_reference)
            img_reference.save_image(image)

    def get_image_path(self, name):
        """Returns the path to serve statically of a file saved with :func:`tropofy.app.AppDataSet.save_image`

        This function is usually used with :class:`tropofy.app.StaticImage`.

        :param name: Name of file (must match the name of an image saved with :func:`tropofy.app.AppDataSet.save_image`)
        :type name: string
        :rtype: string

        For example, for a file called 'output.png' saved with :func:`tropofy.app.AppDataSet.save_image`:

        .. code-block:: python

           data_set.get_image_path('output.png')
        """

        ref = self._get_image_reference_by_name(name)
        return ref.static_path if ref else None

    def _get_image_reference_by_name(self, name):
        return next((ref for ref in self.saved_image_references if ref.image_name == name), None)

    def get_path_of_file_in_data_set_folder(self, file_name):
        """Returns a path on disk, within a folder specific for the data set, that is safe to save the file to."""
        return os.path.join(self.file_save_folder, file_name)

    def get_var(self, name):
        """Get a variable that was set with :func:`tropofy.app.AppDataSet.set_var`"""
        try:
            return self.query(DataSetVar).filter_by(name=name).first().value
        except AttributeError:  # param doesn't exit
            return None

    def set_var(self, name, value):
        """Set a variable that can be accessed with :func:`tropofy.app.AppDataSet.get_var`"""
        var = self.query(DataSetVar).filter_by(name=name).first()
        if var:
            var.value = value
        else:
            self.add(DataSetVar(name=name, value=value)) 

    def get_param(self, name):
        """Return the value of a :class:`tropofy.app.Parameter`

        :param name: Name of parameter. Must match a parameter name defined in :func:`tropofy.app.AppWithDataSets.get_parameters`
        :type name: str

        :returns: Parameter value
        """
        return self.get_var(name)

    def set_param(self, name, value):
        """Set the value of a :class:`tropofy.app.Parameter`

        :param name: Name of parameter. Must match a parameter name defined in :func:`tropofy.app.AppWithDataSets.get_parameters`
        :type name: str
        :param value: Value of parameter. Must satisfy any value restrictions specified on the parameter.
        """
        param = self.app.get_param(name)
        if param:
            if param.allowed_type:  # Will check correct type. Will also convert to correct type if able.
                value = Validator.validate(value, param.allowed_type, name)
            if param.validator:
                validation = param.validator(value)
                if validation != True:
                    raise ValidationException(name=name, value=value, message=validation)
            self.set_var(name, value)
        else:
            raise ValidationException(name=name, value=value, message='Parameter does not exist.')
        

    def init_parameters_with_defaults(self):  # Init parameter defaults on data set creation
        for param in self.app.get_parameters():
            self.set_param(name=param.name, value=param.default)


    __mapper_args__ = {   # Todo: Remove these? No inheritance appears to be happening so not needed...
        'polymorphic_on': type_str,
        'polymorphic_identity': 'AppDataSet'
    }


class DataSetVar(DataSetMixin):
    name = Column(Text, nullable=False)
    value_as_json_str = Column(Text, nullable=True)
    stored_type = Column(Text)  # Stores the type of object that got stored, so that it can be decoded back out. 

    __table_args__ = (
        UniqueConstraint('name', 'data_set_id'),
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {}
    )

    def __init__(self, name, value=None, type=None):
        self.name = name
        self.value = value
        
    @property
    def value(self):        
        json_str = json.loads(self.value_as_json_str)
        return Validator.validate(json_str, self.stored_type, self.name)

    @value.setter
    def value(self, value):
        try:
            self.value_as_json_str = json.dumps(value, cls=ExtendedJsonEncoder)
            self.stored_type = type(value).__name__
        except TypeError:
            raise
        


class FunctionExecutionInfo(DataSetMixin):
    widget_id = Column(Integer, nullable=False)
    is_currently_running = Column(Boolean, nullable=False, default=False)
    last_run_messages = Column(Text, nullable=False, default="")
    num_times_run = Column(Integer, nullable=False, default=0)
    last_run_start_time = Column(DateTime)

    __table_args__ = (
        UniqueConstraint('widget_id', 'data_set_id'),
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {}
    )

    def __init__(self, widget_id):
        self.widget_id = widget_id

    def record_execution_start(self):
        self.is_currently_running = True
        self.last_run_start_time = datetime.now() + timedelta(hours=10)  # TODO: AWS Machine is UTC. Make Brisbane time for now. Ideally record from client side JS and get user local times.
        self.last_run_messages = ""
        self.num_times_run += 1

    @property
    def running_duration_str(self):
        s = ((datetime.now() + timedelta(hours=10))-self.last_run_start_time).seconds  # TODO: Remove the +10 hrs. Times should be on client side.
        hours, remainder = divmod(s, 3600)
        minutes, seconds = divmod(remainder, 60)
        return '%s:%s:%s' % (hours, minutes, seconds)

    def update_msgs(self, msg):
        self.last_run_messages += "<br>%s" % msg

    def update_msgs_wo_orm(self, msg, engine):  # Commits directly to DB. Slows things down quite a lot!!!
        function_execution_info_table = self.metadata.tables['functionexecutioninfo']
        update_statement = function_execution_info_table.update().where(
            function_execution_info_table.c.id == self.id
        ).values(
            last_run_messages="%s<br>%s" % (self.last_run_messages, msg)
        )
        try:
            engine.connect().execute(update_statement)
        except OperationalError:  # Database was locked - just write it normally to ORM. Won't display in progress dialog though.
            self.last_run_messages += "<br>%s" % msg            


class SavedImageReference(DataSetMixin):
    image_name = Column(Text, nullable=False)
    file_path = Column(Text)
    datetime_saved = Column(DateTime)
    data_set = relationship('AppDataSet')

    __table_args__ = (
        UniqueConstraint('image_name', 'data_set_id'),
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {}
    )

    def __init__(self, image_name):
        self.image_name = image_name

    def save_image(self, image):
        folder = self.data_set.file_save_folder
        self.delete_referenced_file()  # Delete if it exists
        if not os.path.exists(folder):
            os.makedirs(folder)

        self.datetime_saved = dt.now()
        self.file_path = os.path.join(
            folder,
            str(int(time.mktime(self.datetime_saved.timetuple()))) + self.image_name
        )

        image.save(self.file_path)

    @property
    def static_path(self):
        request = threadlocal.get_current_request()
        app = request.app  # static_path is of folder apps_static/app_url_name/data_set_x/filename
        return os.path.join(
            'apps_static',
            app.url_name,
            os.path.relpath(self.file_path, request.app.app_folder_path)
        )

    def delete_referenced_file(self):
        try:  # Remove old file if it exists
            os.remove(self.file_path)
        except (TypeError, OSError):
            pass


@event.listens_for(AppDataSet, 'before_delete')
def receive_before_delete(mapper, connection, data_set):
    try:
        shutil.rmtree(data_set.file_save_folder)
    except OSError:
        pass
