'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from datetime import datetime, date, time
from sqlalchemy.exc import IntegrityError
from tropofy.database.tropofy_orm import ValidationException, Validator
from tropofy.widgets.widgets import Widget


class Form(Widget):
    """A form through which data can be added and edited."""

    class FormElementException(Exception):
        pass

    class Element(object):
        """An element of a form.

        :param name: Form element name. Used to reference the value of the element on submit.
        :type name: str
        :param input_type: Type of form input. ('text', 'select', 'date', 'time') 
        :type input_type: str
        :param default: Default input value. If 'options' provided, default must be an option.
        :type default: Multiple possible types. Must be valid relative to options list and input_type.
        :param label: (optional) Label of form element.
        :type label: str
        :param hidden: (optional) If true, form element will be hidden
        :type hidden: bool
        :param disabled: (optional) If true, form element will be disabled
        :type disabled: bool
        :param options: (optional - used if input_type=='select') Valid options that this element may take.
        :type options: list of strings or list of :class:`tropofy.widgets.Form.Element.Option`
        :param compute_options_from: (optional - used if input_type=='select' and options=None). String matching the `name` of a dependent_option_list of a :class:`tropofy.Form.Element.Option` within a :class:`tropofy.Form.Element` in this widget.
        :type compute_options_from: str
        """
        TEXT = 'text'
        SELECT = 'select'
        DATE = 'date'
        TIME = 'time'

        valid_input_types = ['text', 'select', 'date', 'time']

        def __init__(self, name, input_type, default, label=None, hidden=False, disabled=False, options=None, compute_options_from=None):
            self.name = name

            # Validation
            if input_type not in Form.Element.valid_input_types:
                Form.FormElementException("Invalid input type '%s'. Must be one of %s" % (input_type, str(Form.Element.valid_input_types)))            

            if options and default not in options:
                Form.FormElementException("Default value '%s', must be one of %s" % (default, str(options)))

            self.input_type = input_type
            self.default_value = default
            self.hidden = hidden
            self.disabled = disabled
            self.label = label if label is not None else self.name

            self.options = self._process_options(options) if options else []  # With _process_options, allow old types for backward compatibility (would change app code)
            self.compute_options_from = compute_options_from if self.options == [] and self.is_select else None

        def serialise(self):
            return {
                'name': self.name,
                'inputType': self.input_type,
                'defaultValue': self.default_value,
                'options': [o.__json__() for o in self.options],
                'computeOptionsFrom': self.compute_options_from,
                'label': self.label,
                'hidden': self.hidden,
                'disabled': self.disabled,
            }

        @property
        def is_select(self):
            return self.input_type == 'select'

        def _process_options(self, options):
            """Convert to list of :class:`tropofy.widgets.Form.Element.Option` from option formats:

            Options allowed to be input as list of strings or list of dictionaries - {value: <value>, text: <text>},
            where <text> is the text displayed in the select and <value> the value submitted with the form.

            :rtype: list of :class:`tropofy.widgets.Form.Element.Option`
            """
            converted_options = []
            for option in options:
                if isinstance(option, Form.Element.Option):
                    converted_options.append(option)
                else:
                    if isinstance(option, basestring):
                        option = {'value': option}
                    try:
                        converted_options.append(Form.Element.Option(**option))
                    except:
                        raise Exception('Could not create Form.Element.Option from %s. You should create Form.Element.Option objects directly in your app. Refer to docs for details.' % option)

            return converted_options

        class Option(object):
            """An option of a form select.

            :param value: The value of this option submitted with the form if selected.
            :type value: jsonifiable type (usually str or int)
            :param text: (Optional) Option text displayed in select list. Default to value if not provided.
            :type text: str
            :param dependent_option_lists:
            :type dependent_option_lists: list of dicts of form {name: <option list name> options: <list of :class:`tropofy.widgets.Form.Element.Option`>
            """
            def __init__(self, value, text=None, dependent_option_lists=None):
                self.value = value
                self.text = text if text else self.value
                self.dependent_option_lists = dependent_option_lists if dependent_option_lists else {}

            def __json__(self):
                return {
                    'value': self.value,
                    'text': self.text,
                    'dependentOptionLists': [{
                        'name': name,
                        'options': [o.__json__() for o in options]
                    } for name, options in self.dependent_option_lists.iteritems()]
                }

    def get_type(self):
        return "FormWidget"

    def refresh_from_db(self, data_set, request):
        form_elements = self.get_form_elements(data_set)
        form_elements = form_elements if form_elements else []

        return {
            'formElements': [e.serialise() for e in form_elements],
            'formTitle': self.get_title(data_set),
            'autoSubmit': self.is_autosubmit(data_set),
            'autoSubmitEntireForm': self.autosubmit_entire_form(data_set),
            'hideSubmitButton': self.hide_submit_button(data_set),
        }

    def get_form_elements(self, data_set):
        """Return a list of elements to use in the form.

        :rtype: list of :class:`tropofy.widgets.Form.Element`
        """
        raise NotImplementedError

    def process_data(self, data, data_set):
        """Process data from form submit.

        :param data: Dict of name:value pairs, submitted from the form.
        :type data: dict
        :param data_set: current data set.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :returns: Dictionary in the following form - {'success': <bool>, 'message': <HTML string>, 'results': <results_list>}

            - success (optional bool): Indicates full form submit success or failure.
            - message (optional HTML string): Response message for full form submit.
            - results_list (optional list of dicts of the form): {'name': name, 'success': <bool>, 'message': <str>}.
               - name (str): Must match the name of a corresponding :class:`tropofy.widgets.Form.Element`.
               - success (optional bool): defaults to true. Indicates single form element submit success or failure.
               - message (optional str): default to null. Response message for single form element submit.

        'name', 'success', 'message', 'value'
        :rtype: dict
        """  # Todo: Probably add another element in returns for 'edited_row' so can integrate with Grid.
        raise NotImplementedError

    def get_title(self, data_set):
        """A title for the form. If not implemented then there will be no title.

        :param data_set: current data set.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :returns: Title for the form.
        :rtype: str
        """
        return None

    def is_autosubmit(self, data_set):
        """Determines whether a form autosubmits on value change or has a submit button.

        - True (default): The entire form will autosubmit when any value is changed. No submit button will be shown.
        - False: A submit button will be shown and the form will submit only when the button is clicked.

        :param data_set: current data set.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :returns: Boolean determining if the form will autosubmit.
        :rtype: bool
        """
        return True

    def autosubmit_entire_form(self, data_set):  # Todo: Merge with is_autosubmit to return dict of options.
        """Determines whether a form on autosubmits submits the entire form. Does nothing if :func:`tropofy.widgets.Form.is_autosubmit` returns False.

        Default False - means only form element that has changed is submitted.
        :param data_set: current data set.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :returns: Boolean determining if autosubmit form with submit entire form.
        :rtype: bool
        """
        return False

    def hide_submit_button(self, data_set):
        """Hides the form submit button. Advanced use only - it is usually safe to rely on default value.

        By default, submit button is hidden if :func:`tropofy.widgets.Form.is_autosubmit` returns True, and False otherwise.

        :param data_set: current data set.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :returns: Boolean determining if the form submit button is hidden.
        :rtype: bool
        """
        return self.is_autosubmit(data_set)

    def update_to_db(self, data, data_set_id, oper, data_set):
        form_elements = self.get_form_elements(data_set)
        for element in [e for e in form_elements if data.get(e.name)]:
            if element.input_type == Form.Element.DATE:
                data[element.name] = Validator.validate(data[element.name], date, element.name)
            elif element.input_type == Form.Element.TIME:
                data[element.name] = Validator.validate(data[element.name], time, element.name)

        results = self.process_data(data, data_set)
        return results if results else ""

    @classmethod
    def get_input_type_from_python_type(cls, input_type):
        """Helper function to match python types with form input type.

        Useful in :func:`tropofy.widgets.Form.get_form_elements`, when creating :class:`tropofy.widgets.Form.Element`

        :param input_type: A Python type (eg. int, str, float, bool, date, time)
        :type input_type: type
        :returns: A form input type. One of 'text', 'select', 'data', 'time'.
        :rtype: str
        """
        type_to_form_type = {
            int: 'text',
            str: 'text',
            float: 'text',
            bool: 'select',
            date: 'date',
            time: 'time',
            datetime: 'datetime',
        }
        return type_to_form_type.get(input_type)


class ParameterForm(Form):
    """An implementation of :class:`tropofy.widgets.Form` that displays all :class:`tropofy.app.Parameter` defined in :func:`tropofy.app.AppWithDataSets.get_parameters`

    :param parameter_names_filter: (Optional) List of parameter names to display on form. If none are specified, default is to display all parameters.
    :type parameter_names_filter: list
    """
    def __init__(self, title='Parameters', parameter_names_filter=None):
        self.parameter_names_filter = parameter_names_filter if parameter_names_filter is not None else []
        self.title = title

    def get_form_elements(self, data_set):
        elements = []
        for param in data_set.app.get_parameters():
            if not self.parameter_names_filter or param.name in self.parameter_names_filter:
                value = data_set.get_param(param.name)

                options = param.options
                if param.allowed_type is bool:
                    options = [
                        Form.Element.Option(value=True, text='True'),
                        Form.Element.Option(value=False, text='False'),
                    ]
                    value = 1 if value else 0

                elements.append(Form.Element(
                    name=param.name,
                    input_type='select' if options else ParameterForm.get_input_type_from_python_type(type(value)),
                    default=value,
                    label=param.label,
                    options=options,
                ))
        return elements

    def get_title(self, data_set):
        return self.title

    def process_data(self, data, data_set):
        results = []
        for name, value in data.iteritems():
            try:
                data_set.set_param(name=name, value=value)
            except (ValidationException, IntegrityError) as e:
                results.append({
                    'name': name,
                    'success': False,
                    'message': e.message if e.message else 'A validation error occurred.'
                })
        return {'results': results}


class Filter(Form):
    """An implementation of :class:`tropofy.widgets.Form` specifically designed to act as a filter for other widgets.

    Only the method :func:`tropofy.widgets.Filter.get_form_elements` needs to be implemented for this widget.

    Pass as the first parameter to a :class:`tropofy.widgets.FilteredChart` to link the filter.
    """

    def get_form_elements(self, data_set):
        """Return a list of :class:`tropofy.widgets.Form.Element`

        When implementing :func:`tropofy.widgets.Form.get_form_elements`, ensure the :func:`tropofy.widgets.Form.Element` default values are that which you want your
        filtered widget to use on first use.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: list of :class:`tropofy.widgets.Form.Element`
        """
        raise NotImplementedError

    @property
    def var_name(self):
        return 'filter.%s' % self.id

    def get_values(self, data_set):
        """Values of filter, indexed by form element name.

        It is possible to overide this method and return custom, computed values. For example, if the form has separate input date, and time fields, these could be combined for a datetime value.
        If doing this, it is requiered that all form values be returned also, and not replaced. This is necessary to ensure the form updates on reload appropriately.

        :returns: dict
        """
        values = {}
        for element in self.get_form_elements(data_set):  # Would be a way to make this faster by storing just the names somewhere other than get_form_elements. Would probably need an extra use fn though.
            value = data_set.get_var(self._get_var_name(element.name))  # Don't return a None value, just don't return the key!
            if value is not None:
                values.update({
                    element.name: value
                })
        return values

    def process_data(self, data, data_set):
        for name, value in data.items():
            data_set.set_var(self._get_var_name(name), value)

    def _get_var_name(self, element_name):
        return 'filter_%s_%s' % (self.id, element_name)

    def autosubmit_entire_form(self, data_set):
        return True

    def refresh_from_db(self, data_set, request):
        if not self.get_values(data_set):
            self._set_filter_var_to_defaults(data_set)

        form_element_values = self.get_values(data_set)
        form_elements = self.get_form_elements(data_set)
        for element in form_elements:
            element.default_value = form_element_values[element.name]

        refresh_dict = super(Filter, self).refresh_from_db(data_set, request)
        refresh_dict['formElements'] = [e.serialise() for e in form_elements]  # Have to overide here, as only other option would be to pass form_elements to refresh_from_db -> This is a framework function though, so this would be weird...
        return refresh_dict

    def _set_filter_var_to_defaults(self, data_set):
        # Need to set var to defaults on first load. Otherwise filtered widget will try and load and var will be empty.
        defaults = dict((fe.name, fe.default_value) for fe in self.get_form_elements(data_set))
        self.process_data(
            data=defaults,
            data_set=data_set
        )
