'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import os
import shutil
import tropofy
import pkgutil
import sys
import fileinput


working_dir = os.getcwd()
ini_loc = working_dir + "/local.ini"
app_loc = working_dir + "/apps"


example_apps_folder = '/'.join(tropofy.__file__.replace('\\', '/').split('/')[:-2]) + '/tropofy_example_apps'


def app_not_found_msg(app_name):
    print ("""
        Error: App '{app_name}' not found. Try one of the follow simple Python examples:

        - starter
        - hello_world
        - facility_location
        - batch_geocoding
        - election_results_dashboard
        - generate_kml

        For more examples and detailed install instructions for each, visit http://www.tropofy.com/docs.
    """.format(app_name=app_name))
    sys.exit(1)


def directory_already_exists_msg(directory):
    print (
        "\nError: directory '{directory}' already exists. Use a different directory name, or delete the directory and try again.\n"
    ).format(directory=directory)
    sys.exit(1)


def usage(argv):
    cmd = os.path.basename(argv[0])
    print('\nUsage: %s <directory> <starting_app_name> <developer_key>\n'
          '(example: "%s my_app facility_location 123456")\n\n'

          "Don't have a key? Visit www.tropofy.com to sign up as a developer and get a free key immediately.\n"
          % (cmd, cmd))
    sys.exit(1)


def main(argv=sys.argv):
    """Quickstarts a Tropofy project

    - Creates a directory.
    - Copies a starting_app_file into it
    - Copies a .ini file into it and files out the license_key, apps_directory, database folder.

    """
    if len(argv) != 4:
        usage(argv)

    directory = argv[1]
    app_name = argv[2]
    license_key = argv[3]
    ini_file_location = '/'.join(tropofy.__file__.replace('\\', '/').split('/')[:-1]) + '/ini_files/for_dist/development.ini'

    # Find app folder
    directory_full_path = os.path.join(os.getcwd(), directory)
    app_folder = search_for_subfolder(folder_name=app_name, directory=example_apps_folder)
    if not app_folder:
        app_not_found_msg(app_name)

    # (2) Copy app files
    try:
        shutil.copytree(app_folder, directory_full_path)
    except IOError:
        app_not_found_msg(app_name)
    except OSError:
        directory_already_exists_msg(directory_full_path)

    # (3) Copy ini file
    try:
        shutil.copy(ini_file_location, directory_full_path)
    except IOError:
        print('\nError quickstarting app. Cannot find .ini file. Tropofy installation may need to be reinstalled (pip install tropofy).\n')
        sys.exit(1)

    for line in fileinput.input(os.path.join(directory_full_path, 'development.ini'), inplace=True):
        if line.startswith('custom.apps_directory'):
            print 'custom.apps_directory = ' + directory_full_path
        elif line.startswith('custom.license_key'):
            print 'custom.license_key = ' + license_key
        else:
            print line,

    print(
        '\nSuccess! What to do next:\n'
        'cd ' + directory + '\n'
        'tropofy_run\n'
    )


def search_for_subfolder(folder_name, directory):
    """Returns absolute path of folder_name if found in search_folder"""
    if directory.endswith(folder_name):
        return directory
    else:
        for subfolder in [name for name in os.listdir(directory) if os.path.isdir(os.path.join(directory, name))]:
            search_result = search_for_subfolder(folder_name, os.path.join(directory, subfolder))
            if search_result:
                return search_result
