'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''
from operator import attrgetter

import openpyxl
from pyramid import threadlocal
from openpyxl.writer.excel import save_virtual_workbook
import datetime
import xlrd
import transaction
import collections

from tropofy.file_io.class_tabular_data_mapping import ClassTabularDataMapping

# http://pythonhosted.org/openpyxl/usage.html
# http://pythonhosted.org/openpyxl/api.html
from tropofy.file_io.exceptions import TropofyFileImportExportException


class ClassWsMapping(ClassTabularDataMapping):
    """Maps a Python class to a Microsoft Excel worksheet.

    Used to both write to and read from Microsoft Excel with :func:`tropofy.file_io.read_write_xl.ExcelReader` and :func:`tropofy.file_io.read_write_xl.ExcelWriter`.

    :param class\_: Python class in mapping.
    :type class\_: class
    :param ws_name: Excel worksheet name in mapping.
    :type ws_name: str
    :param attribute_column_aliases: dict of class attribute to col alias.
    :type attribute_column_aliases: dict
    :param process_objects: (Optional) Function that processes the objects of class\_ loaded from Excel. Must accept a single parameter which will be passed a list of objects of class\_.
       If no function is passed, default behaviour is to assume class\_ is a SQLA class, and attempt to write the objects to the database.
    :type process_objects: function
    :param get_objects: (Optional - default :func:`tropofy.file_io.ClassTabularDataMapping.get_all_sqla_objects_of_class_in_data_set`)
     A function that accepts ``data_set`` and ``class_`` as parameters and returns a list of objects of ``class_``.
    :type get_objects: function
    :param objects: (Optional - default None) Specify objects instead of dynamically generating them with ``get_objects``.
     A list of objects of ``class_`` to be included in export. Will overide ``get_objects`` if value is not None.
    :type objects: list

    .. note:: To specify the order in which columns are written, use a ``collections.OrderedDict`` (Python built-in) for ``attribute_column_aliases``.
    """

    def __init__(self, class_, ws_name, attribute_column_aliases, process_objects=None, get_objects=None, objects=None):
        self.ws_name = ws_name
        super(ClassWsMapping, self).__init__(class_, attribute_column_aliases, process_objects, get_objects, objects)

    @classmethod
    def create_ordered_mappings_from_sqla_classes(cls, sqla_classes=None, include_parameters_if_any=True, app=None):
        """
        :param sqla_classes: List of SQLA classes to create mappings for. If None, create for all classes in app.
        :type sqla_classes: list
        :param app: (Default: App from which this function is called) App in which sqla classes are defined.
        :type app: :class:`tropofy.app.App`
        :returns: list of :class:`tropofy.database.read_write_xl.ClassWsMapping`
        :rtype: list
        """
        sqla_classes = sqla_classes if sqla_classes is not None else []
        if not isinstance(sqla_classes, collections.Iterable):
            raise TypeError("param sqla_classes must be an iterable (e.g. list)")

        app = app if app is not None else threadlocal.get_current_request().app
        class_ws_mappings = []
        if app:
            tables_mapped_to_classes = app.get_dependency_ordered_dict_of_tables_mapped_to_classes_for_app()
            for table, class_ in tables_mapped_to_classes.iteritems():
                if sqla_classes and class_ not in sqla_classes:  # Map all classes if none specified. Also only map classes in sqla_classes
                    class_ = None

                if class_:
                    required_attribute_names = [col.name for col in table.columns if col.name not in ['id', 'data_set_id']]
                    attribute_column_aliases = collections.OrderedDict()  # Make the column ordering in the excel file the same as that in the grids
                    for name in required_attribute_names:
                        attribute_column_aliases.update({name: name})

                    class_ws_mappings.append(cls(
                        class_=tables_mapped_to_classes[table],
                        ws_name=table.name,
                        attribute_column_aliases=attribute_column_aliases,
                        process_objects=cls.process_sqla_objects,
                        get_objects=cls.get_all_sqla_objects_of_class_in_data_set,
                    ))

        class ParamFromExcel(object):
            def __init__(self, name, value):
                self.name = name
                self.value = value

            @classmethod
            def load_params_into_data_set(cls, data_set, params):
                for param in params:
                    data_set.set_param(
                        name=param.name,
                        value=param.value,
                    )

            @classmethod
            def get_params_from_data_set(cls, data_set, class_):
                params = []
                for param in data_set.app.get_parameters():
                    params.append(class_(
                        name=param.name,
                        value=data_set.get_param(param.name)
                    ))
                return params

        if include_parameters_if_any and app.get_parameters():
            class_ws_mappings.append(cls(
                class_=ParamFromExcel,
                ws_name='parameters',
                attribute_column_aliases={
                    'name': 'name',
                    'value': 'value'
                },
                process_objects=ParamFromExcel.load_params_into_data_set,
                get_objects=ParamFromExcel.get_params_from_data_set,
            ))
        return class_ws_mappings

    def ws_has_required_columns(self, ws):
        return self.tabular_data_has_required_columns(
            tabular_data_column_names_set=_ws_headers_as_set(ws),
            data_container_name=self.ws_name
        )


class ExcelWriter(object):
    """Utility class for writing data to Microsoft Excel files."""
    @classmethod
    def create_excel_string_repr(cls, data_set, ordered_class_ws_mappings=None):
        """
        Create an Excel file as a string, which has a worksheet for each class in ordered_class_ws_mappings.

        Can be used as return value of :func:`tropofy.widgets.custom_export_widget.get_custom_export_data`.

        If ``ordered_class_ws_mappings`` is not passed, a 'full export' of all data in the data set will be made, with one worksheet made for each SQLA ORM class.

        If no ``objects`` or ``get_objects`` param is passed to a ``ordered_class_ws_mapping``, it is assumed that ``ordered_class_ws_mapping.class_`` is a SQLA ORM
        class, and all objects in the database for this ``data_set`` will be written to Excel.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param ordered_class_ws_mappings: (Optional) Default None. If None is passed, automatic mappings will be made to do a 'full export' of all data in all SQLA ORM classes in the app.
        :type ordered_class_ws_mappings: list of :class:`tropofy.file_io.read_write_xl.ClassWsMapping`
        :return: str
        """
        if ordered_class_ws_mappings is None:
            ordered_class_ws_mappings = ClassWsMapping.create_ordered_mappings_from_sqla_classes()
        for m in ordered_class_ws_mappings:
            m.objects = m.objects if m.objects is not None else m.get_objects(data_set, m.class_)
        wb = cls._create_excel_wb_from_class_mappings(ordered_class_ws_mappings)
        return save_virtual_workbook(wb)

    @classmethod
    def create_excel_string_repr_from_sqla_classes(cls, data_set, sqla_classes):
        """
        Create an Excel file as a string, which has a worksheet for each SQLA class passed with SQLA_classes.

        Can be used as return value of :func:`tropofy.widgets.custom_export_widget.get_custom_export_data`.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param sqla_classes: A list of SQLA ORM defined classes
        :type sqla_classes: list of SQLA classes
        :return: str
        """
        wb = cls._create_excel_wb_from_sqla_classes(data_set, sqla_classes)
        return save_virtual_workbook(wb)

    @classmethod
    def _create_excel_wb_from_sqla_classes(cls, data_set, sqla_classes):
        """Create an openpyxl Workbook, which has a tab for each SQLA class passed with SQLA_classes.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param sqla_classes: A list of SQLA ORM defined classes
        :type sqla_classes: list of SQLA classes
        :return: openpyxl.Workbook
        """
        ordered_class_ws_mappings = ClassWsMapping.create_ordered_mappings_from_sqla_classes(sqla_classes=sqla_classes, include_parameters_if_any=False)
        for m in ordered_class_ws_mappings:
            m.objects = m.get_objects(data_set, m.class_)
            m.objects.sort(key=attrgetter(*m.attribute_column_aliases.keys()))
        return cls._create_excel_wb_from_class_mappings(ordered_class_ws_mappings=ordered_class_ws_mappings)

    @classmethod
    def _create_excel_wb_from_class_mappings(cls, ordered_class_ws_mappings):
        """Create an openpyxl in memory Excel workbook from a list of :class:`tropofy.file_io.read_write_xl.ClassWsMapping`.

        :param ordered_class_ws_mappings: Ordered list of :class:`tropofy.file_io.read_write_xl.ClassWsMapping`. Will be processed in this order.
        :type ordered_class_ws_mappings: ClassWsMapping list
        """
        wb = openpyxl.Workbook()
        if wb.get_sheet_by_name('Sheet') is not None:
            wb.remove_sheet(wb.get_sheet_by_name('Sheet'))

        for mapping in ordered_class_ws_mappings:
            worksheet = wb.create_sheet()
            try:
                worksheet.title = mapping.ws_name[:29]  # Excel ws names have a limit of 31 characters. Restrict to first 29 to allow excel to append numbers if names aren't unique.
            except openpyxl.shared.exc.SheetTitleException:
                raise openpyxl.shared.exc.SheetTitleException("Could not create > 100 worksheets with names with duplicate first 29 characters.")
            col_names = mapping.required_column_names
            if col_names:
                for col in col_names:
                    worksheet.cell(row=0, column=col_names.index(col)).value = col

                for i, obj in enumerate(mapping.objects):
                    for j, col_name in enumerate(col_names):
                        try:
                            value = getattr(obj, mapping.get_attribute_from_column_name(col_name))
                        except AttributeError:
                            print('Object in ClassWsMappingForExport.objects does not match ClassWsMappingForExport object and col definition')
                            raise
                        row_num = i + 1  # Add one to row to skip col names header row.
                        if isinstance(value, basestring):  # Have to force string. Otherwise will automatically coerce to things. Eg scientific notation: '110E04' will save as 1100000.0
                            worksheet.cell(row=row_num, column=j).set_value_explicit(
                                value=value,
                                data_type=openpyxl.cell.Cell.TYPE_STRING
                            )
                        else:
                            worksheet.cell(row=row_num, column=j).value = value  # Add one to row to skip col names header row.
        return wb


class ExcelReader(object):
    """Utility class for reading data from Microsoft Excel files. Supports .xlsx, .xls, .xlsm formats.

    Each :class:`tropofy.file_io.read_write_xl.ClassWsMapping` maps a worksheet in the Excel file to a Python class.
    These method will attempt to create objects of each :class:`tropofy.file_io.read_write_xl.ClassWsMapping` class from the
    corresponding mapping worksheet, and then pass them to the process_objects function of the mapping. In process_objects,
    the objects can be transformed in any way, or left alone, and then finally some data would be written to the database to make it persist.

    Any :class:`tropofy.file_io.read_write_xl.ClassWsMapping` that does not map to a worksheets in the Excel file will be ignored. A message will report
    that no instances of the class were loaded.

    If ordered_class_ws_mappings is not passed, a list of generic mappings for each SQLA ORM class will be used. For a given SQLA ORM class,
    a worksheet is looked for with the lowercase name of the class, with headings matching the defined columns.

    To import data not in a format that maps each column to an attribute of a class, use another Python class that does map to the format, and then
    process these into the class you need with the process_objects function.
    """

    @classmethod
    def load_data_from_excel_file_on_disk(cls, data_set, file_path, ordered_class_ws_mappings=None):
        """Load data from an Excel file on disk into the app.

        To load data from an Excel workbook that contains worksheets matching some or all of the SQLA ORM classes in an app:

        .. literalinclude:: ../../../tropofy_example_apps/core_dependencies/facility_location/facility_location.py
           :pyobject: load_brisbane_data
           :start-after: def

        The following example shows how to import data from an Excel file located in the base directory of your app module. A SQLA ORM class ``Location`` exists
        with parameters ``name``, ``lat``, and ``lng``. The Excel workbook located in the base directory of app module is called 'locations.xlsx' and contains
        a worksheet named 'Locations'. This worksheet has three columns with headings 'Name', 'Latitude', and 'Longitude. Parameter ``attribute_column_aliases``
        of :class:`tropofy.file_io.read_write_xl.ClassWsMapping` describes the mapping of column names to the parameters of ``Location``.
        As no ``process_objects`` function parameter has been supplied, :class:`tropofy.file_io.read_write_xl.ExcelReader` will by default assume ``Location``
        is a SQLA ORM class and save each object read from the worksheet to the database.

        .. code-block:: python

            read_write_xl.ExcelReader.load_data_from_excel_file_on_disk(
                data_set,
                data_set.app.get_path_of_file_in_app_folder('locations.xlsx'),
                ordered_class_ws_mappings=[read_write_xl.ClassWsMapping(
                    class_=Location,
                    ws_name="Locations",
                    attribute_column_aliases={
                        'name': 'Name',
                        'lat': 'Latitude',
                        'lng': 'Longitude',
                    },
                )]
            )

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param file_path: Path on disk of the Excel file.
        :type file_path: str
        :param ordered_class_ws_mappings: (Optional) Default None. If None is passed, automatic mappings will be made to do a 'full import' of all data in all SQLA ORM classes in the app.
        :type ordered_class_ws_mappings: list of :class:`tropofy.file_io.read_write_xl.ClassWsMapping`
        :return: str
        """
        wb = xlrd.open_workbook(file_path)
        cls._load_data_from_excel(wb, data_set, ordered_class_ws_mappings)

    @classmethod
    def load_data_from_excel_file_in_memory(cls, data_set, file_in_memory, ordered_class_ws_mappings=None):
        """Load data from an Excel file in memory into the app.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param file_in_memory: In memory representation of an Excel file. Likely uploaded through a `tropofy.widgets.FileUpload` widget. Can be a string repr or FieldStorage instance.
         If processing a FieldStorage instance, the file cannot be processed more than once. Turn the file into a basestring before calling this method to process the file more than once.
        :type file_in_memory: basestring or FieldStorage.
        :param ordered_class_ws_mappings: (Optional) Default None. If None is passed, automatic mappings will be made to do a 'full import' of all data in all SQLA ORM classes in the app.
        :type ordered_class_ws_mappings: list of :class:`tropofy.file_io.read_write_xl.ClassWsMapping`
        :return: str
        """
        if not isinstance(file_in_memory, basestring):
            try:
                file_in_memory = file_in_memory.file.read()
            except:
                pass

        try:
            wb = xlrd.open_workbook(file_contents=file_in_memory)
        except:
            raise Exception('Uploaded file could not be read.')
        return cls._load_data_from_excel(wb, data_set, ordered_class_ws_mappings)

    @classmethod
    def _load_data_from_excel(cls, wb, data_set, ordered_class_ws_mappings=None):
        """
        :param wb: Excel workbook opened with xlrd
        :param ordered_class_ws_mappings: Ordered list of :class:`tropofy.file_io.read_write_xl.ClassWsMapping`. Will be processed in this order (makes safe for dependencies)
        :type ordered_class_ws_mappings: list of :class:`tropofy.file_io.read_write_xl.ClassWsMapping`
        """
        if not ordered_class_ws_mappings:
            ordered_class_ws_mappings = ClassWsMapping.create_ordered_mappings_from_sqla_classes()

        progress_messages = []
        for mapping in ordered_class_ws_mappings:
            try:
                if mapping.ws_name in wb.sheet_names():
                    ws = wb.sheet_by_name(mapping.ws_name)
                    if ws and mapping.ws_has_required_columns(ws):
                        headers = ws.row_values(0)
                        new_objects = []
                        curr_row = 0
                        while curr_row < ws.nrows - 1:
                            curr_row += + 1

                            init_args = {}
                            for attr, header in mapping.attribute_column_aliases.iteritems():
                                value = cls._cellval(ws.row(curr_row)[headers.index(header)], wb.datemode)
                                init_args.update({attr: value})

                            new_object = None
                            try:
                                new_object = mapping.class_(**init_args)
                            except Exception as e:  # Probably a ValidationException
                                raise TropofyFileImportExportException([
                                    "There was a problem constructing a '%s' from the data %s." % (mapping.class_name, str(init_args)),
                                    str(e),
                                ])
                            try:
                                new_object.data_set_id = data_set.id
                            except AttributeError:
                                # object doesn't have a data_set_id parameter.
                                # Useful for using namedtuples as import objects that are then processed into objects on the data set in `mapping.process_objects`
                                pass
                            new_objects.append(new_object)

                        try:
                            mapping.process_objects(data_set, new_objects)
                            progress_messages.extend(["Created {num} instances of type {type}".format(num=str(len(new_objects)), type=mapping.class_name)])
                        except Exception as e:
                            raise TropofyFileImportExportException([
                                "There was an error processing {num} instances of type {type}".format(num=str(len(new_objects)), type=mapping.class_name),
                                str(e),
                            ])
                    else:
                        progress_messages.extend(["Worksheet %s does not have columns: %s." % (mapping.ws_name, mapping.required_column_names - _ws_headers_as_set(ws))])
                else:
                    progress_messages.extend(["There is no sheet named %s." % mapping.ws_name])
            except Exception as e:
                progress_messages.extend([
                    "Aborted - Data set deleted.",
                    "There was a problem with sheet <{sheet_name}>: ".format(sheet_name=mapping.ws_name),
                ])
                try:  # Get history_as_list of Excel Error if it exists
                    progress_messages.extend(e.history_as_list)
                except AttributeError:
                    progress_messages[-1] += str(e)

                transaction.abort()
                raise TropofyFileImportExportException(progress_messages)
        print("\nLoading data from Excel...")
        for msg in progress_messages:
            print(msg)
        return progress_messages

    @classmethod
    def _cellval(cls, cell, datemode):
        if cell.ctype == xlrd.XL_CELL_DATE:
            datetuple = xlrd.xldate_as_tuple(cell.value, datemode)
            if datetuple[:3] == (0, 0, 0):
                return datetime.time(datetuple[3], datetuple[4], datetuple[5])
            #elif datetuple[3:] == (0, 0, 0):  # Doesn't work for 12:00 am!
            #    return datetime.date(datetuple[0], datetuple[1], datetuple[2])
            else:
                return datetime.datetime(datetuple[0], datetuple[1], datetuple[2], datetuple[3], datetuple[4], datetuple[5])
        if cell.ctype == xlrd.XL_CELL_EMPTY:
            return None
        if cell.ctype == xlrd.XL_CELL_BOOLEAN:
            return cell.value == 1
        return cell.value

# ws helper function
def _ws_headers_as_set(ws):
    headers = ws.row_values(0)
    return set([col.encode('ascii', 'ignore') for col in headers if col is not None])

