'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and governed by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import boto
from boto.s3.key import Key as s3_key
from tropofy.file_io.exceptions import TropofyFileImportExportException
from tropofy.file_io import FileIoConfig


class AwsS3Utility(object):
    """Utility module for interacting with AWS S3.

    .. note:: This is a helper utility module that is only used directly in advanced cases.
       For most use cases, use :class:`tropofy.file_io.read_write_aws_s3.AwsS3Writer` and :class:`tropofy.file_io.read_write_aws_s3.AwsS3Reader`

    To use any of the AwsS3 modules you must:

    - Install Python module `boto <https://pypi.python.org/pypi/boto>`_. From within your virtualenv type: ``pip install boto``.
    - Set the parameter ``custom.aws_s3_bucket`` in your .ini file used to run your Tropofy app.
    - Create an environment var ``AWS_CREDENTIAL_FILE`` which points to a file with your AWS credentials in it. This file must take the form:

        .. code-block:: text

           AWSAccessKeyId=your_aws_access_key
           AWSSecretKey=your_aws_secret_key
    """
    @classmethod
    def get_unique_s3_key(cls, data_set, file_name):
        """Returns a unique s3 key in any bucket given a data_set and file_name.

        Uses data_set.created_datetime to make unique for multiple servers running the same app pointing at the same s3 bucket (e.g. dev and production).
        Will be unique as long as separate servers running the same app do not create data sets with the same id, at the same second.

        :param data_set:
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param file_name: Name of file - not a full s3 key. e.g. `output.xlsx`.
        :type file_name: str
        """
        return "/app_downloads/%s/data_set_%s/%s/%s" % (
            data_set.app.url_name,
            data_set.id,
            data_set.created_datetime.strftime("%Y%m%d-%H:%M:%S"),
            file_name,
        )

    @classmethod
    def get_bucket(cls):
        """Connect to and return the AWS S3 bucket specified in .ini parameter custom.aws_s3_bucket.

        :rtype: boto.s3.bucket.Bucket
        """
        if FileIoConfig.aws_s3_bucket_name is None:
            raise TropofyFileImportExportException("Unable to connect to AWS S3. You must supply an AWS S3 bucket name in .ini parameter custom.aws_s3_bucket")
        s3 = boto.connect_s3()
        return s3.get_bucket(FileIoConfig.aws_s3_bucket_name)


class AwsS3Writer(AwsS3Utility):
    """Utility module for aiding in saving files to `AWS S3 <aws.amazon.com/s3/>`_."""

    @classmethod
    def save_file_string(cls, data_set, file_name, string_data):
        """Save file to AWS s3.

        Uses bucket defined in .ini param custom.aws_s3_bucket and creates a unique key accessible with :func:`tropofy.file_io.read_write_aws_s3.AwsS3Utility.get_unique_s3_key`.

        :param data_set:
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param file_name: Name of file - not a full s3 keyc. e.g. `output.xlsx`.
        :type file_name: str
        :param string_data: Data to save
        :type string_data: str
        """

        k = s3_key(cls.get_bucket())
        k.key = cls.get_unique_s3_key(data_set, file_name)
        k.set_contents_from_string(string_data, replace=True, reduced_redundancy=True)


class AwsS3Reader(AwsS3Utility):
    """Utility module for aiding in reading files to `AWS S3 <aws.amazon.com/s3/>`_."""

    @classmethod
    def get_file_access_url(cls, data_set, file_name, seconds_url_expires_in=60):
        """Returns a url from which a file can be accessed.

        :param data_set:
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param file_name: Name of file - not a full s3 key. e.g. `output.xlsx`.
        :type file_name: str
        :param seconds_url_expires_in: Numbers of seconds the url is valid for.
        :type seconds_url_expires_in: int
        :returns: url string from which the file can be accessed from AWS S3 for the number of seconds specified by ``seconds_url_expires_in``.
        :rtype: str
        """
        k = s3_key(cls.get_bucket(), cls.get_unique_s3_key(data_set, file_name))
        return k.generate_url(expires_in=seconds_url_expires_in)