from celery.signals import celeryd_init, task_prerun, task_postrun, before_task_publish, task_revoked, task_failure
from celery import current_task
from sqlalchemy import create_engine
from sqlalchemy.orm import scoped_session, sessionmaker
from tropofy.database.tropofy_orm import DynamicSchemaName


if DynamicSchemaName.use_app_schemas is None:
    DynamicSchemaName.set_use_app_schemas(True)

from tropofy.app import AppDataSet, CeleryTask, TaskResultMeta # Needs to be after setting .use_app_schemas


# @celeryd_init.connect()
# def configure_worker(conf=None, **kwargs):
#     """Set session that has nothing to do with a web server (db_session for Tropofy usually does)."""
#     AppDataSet.db_session = scoped_session(sessionmaker(autocommit=False, autoflush=False))


@before_task_publish.connect()
def before_task_publish_handler(sender=None, body=None, **kwargs):
    """Strip id and db engine url from AppDataSet for JSON transfer to worker node. Works for data_set in args or kwargs"""
    data_set = None
    try:
        data_set = body['kwargs'].pop('data_set')
        body['kwargs'].update({"__data_set_id": data_set.id})
    except KeyError:
        for i, arg in enumerate(body['args']):
            if type(arg) is AppDataSet:
                data_set = arg  # Can't just pop as body['args'] is a tuple (which means it's immutable).
                args = []
                for a in body['args']:
                    args.append(a if a != data_set else data_set.id)
                body['args'] = args
                body['kwargs'].update({"__data_set_id_arg_index": i})
                break  # If multiple AppDataSet's passed as args, convert the first one only (noone should ever do this?)

    try:
        body['kwargs'].update({
            "__engine_url": str(data_set.app.engine.url),
            "__app_module_path": data_set.app_module_path,
            "__app_url_name": data_set.app_url_name,
        })
    except AttributeError:
        pass  # No data_set found in args or kwargs


@task_prerun.connect()
def task_prerun_handler(sender=None, body=None, **kwargs):
    """Inits the engine connections str, and gets the AppDataSet object from the db."""

    # Initialise _custom_meta property on task. Will store list of meta while the task is running. Include messages, header, etc.
    sender._running_task_meta = TaskResultMeta()

    try:
        engine = create_engine(kwargs['kwargs'].pop('__engine_url'))
        AppDataSet.db_session = scoped_session(sessionmaker(bind=engine, autocommit=False, autoflush=False))
    except KeyError:
        print("""Warning! No database engine_url provided so no connection could be made.
        Provide an AppDataSet arg to a celery task, or the keyed param __engine_url if not using data_sets.
        """)

    data_set = None
    try:  # data_set as arg
        data_set_id_arg_index = kwargs['kwargs'].pop('__data_set_id_arg_index')
        data_set_id = kwargs['args'][data_set_id_arg_index]
        data_set = AppDataSet.db_session().query(AppDataSet).filter_by(id=data_set_id).one()
        kwargs['args'][data_set_id_arg_index] = data_set  # Args is list not tuple as have modified for data_set_id already.
    except KeyError:
        try:  # data_set as kwarg
            data_set = AppDataSet.db_session().query(AppDataSet).filter_by(id=kwargs['kwargs'].pop('__data_set_id')).one()
            kwargs['kwargs'].update({'data_set': data_set})
        except KeyError:
            pass
    if data_set:
        try:
            data_set._app_module_path = kwargs['kwargs'].pop('__app_module_path')
            data_set._app_url_name = kwargs['kwargs'].pop('__app_url_name')
        except KeyError as e:
            raise Exception("""__app_module_path and __app_url_name could not be set on data_set for Celery task. Not being sent through in kwargs correctly.""")
    else:
        print("""Warning! No data set could be created.
        If you want access to data_set, provide an AppDataSet object as a parameter to this celery task.
        """)


@task_postrun.connect()
def task_postrun_handler(sender=None, body=None, **kwargs):
    """Finalises task, ensuring that the connection to the database is closed and commited/rolled back as appropriate on task completion."""
    celery_task = _get_celery_task_from_handler_kwargs(kwargs)
    finished_status = kwargs['state']
    task_successful = finished_status == CeleryTask.SUCCESS

    if not task_successful:
        AppDataSet.db_session.rollback()

    if not celery_task.terminated_by_user:  # Finalises task in :func:`tropofy.app.CeleryTask.terminate` so don't need to do it again.
        task_return_val = kwargs['retval']
        task_meta = current_task._running_task_meta
        if isinstance(task_return_val, Exception):
            task_meta.add_exception_message(task_return_val)
        task_meta.finalise_task(finished_status=kwargs['state'])
        celery_task._finished_result = task_meta

    AppDataSet.db_session.commit()
    AppDataSet.db_session.remove()


def _get_celery_task_from_handler_kwargs(kwargs):
    """Searches kwargs and args for :class:`tropofy.app.AppDataSet`. Then queries is for task"""
    task_id = kwargs['task_id']
    data_set = kwargs['kwargs'].get('data_set')
    if not data_set:
        for i, arg in enumerate(kwargs['args']):
            if type(arg) is AppDataSet:
                data_set = arg
                break
    return data_set.query(CeleryTask).filter_by(task_id=task_id).first()