'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import transaction
from pyramid.config import Configurator
from pyramid.authentication import AuthTktAuthenticationPolicy
from pyramid.authorization import ACLAuthorizationPolicy
from pyramid.session import UnencryptedCookieSessionFactoryConfig
from pyramid.security import Allow, Everyone, Authenticated, authenticated_userid
import pyramid.httpexceptions as exc
from sqlalchemy.orm.exc import MultipleResultsFound, NoResultFound

from tropofy.database import DBSession
from tropofy.database.tropofy_orm import json_renderer, DynamicSchemaName
from tropofy.file_io import FileIoConfig


def main(global_config, **settings):
    print("**************************************************************************************************")
    print("Initialising server...")
    global_sqla_engine = settings.get('sqlalchemy.url', '')

    # If 'custom.use_app_schemas' not specified, use app schemas if a sqlalchemy.url is specified. 
    # (sqlite is used if sqlalchemy.url not specified. Don't use schema's with SQLite!
    DynamicSchemaName.set_use_app_schemas(settings.get('custom.use_app_schemas', bool(global_sqla_engine)))

    # Delay imports, so that DynamicSchemaName.use_app_schemas can be updated correctly before AppDataSet etc. tables are imported.
    from tropofy.app import App, AppDataSet, AppManager, LoadedApps
    from tropofy.widgets import Widget
    App.overridden_global_sqla_engine = global_sqla_engine

    authn_policy = AuthTktAuthenticationPolicy('sosecret5872', callback=groupfinder, hashalg='sha512')
    authz_policy = ACLAuthorizationPolicy()
    my_session_factory = UnencryptedCookieSessionFactoryConfig('itsatropofyseekreet')
    config = Configurator(
        settings=settings,
        root_factory=RootFactory,
        authentication_policy=authn_policy,
        authorization_policy=authz_policy,
        session_factory=my_session_factory
    )

    # Define helper request.* functions
    def get_data_set(request):
        try:
            data_set_id = int(request.matchdict.get('data_set_id'))
            if data_set_id >= 0:
                return DBSession().query(AppDataSet).filter(AppDataSet.id == data_set_id).one()
        except (MultipleResultsFound, NoResultFound):
            return None
        except:
            return None
        return None


    def get_app(request):
        app = AppManager.get_app_from_url_name(request.matchdict.get('app_url_name').lower())
        if app:
            return app
        else:
            raise exc.HTTPNotFound()


    def get_app_if_only_one_app(request):
        """Developer mode serves one app per compute server. This returns that app.
        If more than one app, None is returned.
        Used for redirect of / root to run_app/<app_url_name>.
        """
        return AppManager.get_app_if_only_one()


    def get_data_set_id(request):
        return request.data_set.id if request.data_set else None


    def get_widget(request):
        if request.matchdict.get('widget_id', None):
            return Widget.widget_dict.get(int(request.matchdict['widget_id']), None)
        return None

    config.add_request_method(get_data_set, 'data_set', reify=True)
    config.add_request_method(get_app, 'app', reify=True)
    config.add_request_method(get_app_if_only_one_app, 'only_app', reify=True)
    config.add_request_method(get_widget, 'widget', reify=True)
    config.add_request_method(get_data_set_id, 'data_set_id', reify=True)

    config.add_static_view(name='static', path='static')  # Needed for respond.proxy.gif.
    config.add_renderer('json', json_renderer)  # Custom json renderer.

    config.include(addroutes)
    config.scan(package='tropofy.views')  # Scans for @view_config decorators to register views.

    AppManager.add_additional_absolute_paths_to_python_path(settings.get('custom.additional_absolute_paths_to_add_to_python_path'))
    AppManager.initialise_apps(settings['custom.apps_directory'])
    for app in LoadedApps.apps.values():
        config.add_static_view('apps_static/%s' % app.url_name, app.app_folder_path, cache_max_age=1)

    FileIoConfig.aws_s3_bucket_name = settings.get('custom.aws_s3_bucket')

    transaction.commit()

    if App.overridden_global_sqla_engine:
        print("You are connecting to a specific database: %s\n" % App.overridden_global_sqla_engine)

    app_urls = [app.url_name for app in LoadedApps.apps.values()]
    if len(app_urls) == 1:
        print("**************************************************************************************************")
        print("If you have not already, open up http://localhost:8080/ in your browser to login and view your app")
    else:
        print("Serving multiple apps on paths:")
        for url in app_urls:
            print("/%s" % url)

    print("**************************************************************************************************")
    return config.make_wsgi_app()


def addroutes(config):
    config.add_route('healthcheck', '/healthcheck')
    config.add_route('create_new_data_set', '/{app_url_name}/create_new_data_set', factory='tropofy.AppFactory')
    config.add_route('start_app_with_data_set', '/{app_url_name}/start_app_with_data_set/{data_set_id:-?\d+}', factory='tropofy.AppFactory')
    config.add_route('set_data_set_current_step', '/{app_url_name}/set_data_set_current_step/{current_step_index:\d+}/{data_set_id:-?\d+}', factory='tropofy.AppFactory')
    config.add_route('refresh_widget_from_db', '/{app_url_name}/refresh_widget_from_db/{widget_id:\d+}/{data_set_id:-?\d+}', factory='tropofy.AppFactory')
    config.add_route('update_widget_to_db', '/{app_url_name}/update_widget_to_db/{widget_id:\d+}/{oper}/{data_set_id:-?\d+}', factory='tropofy.AppFactory')
    config.add_route('update_widget_from_db', '/{app_url_name}/update_widget_from_db/{widget_id:\d+}/{oper}/{data_set_id:-?\d+}', factory='tropofy.AppFactory')
    config.add_route('export_data_set', '/{app_url_name}/export_data_set/{data_set_id:-?\d+}', factory='tropofy.AppFactory')
    config.add_route('login', '/{app_url_name}/login')
    config.add_route('logout', '/{app_url_name}/logout')
    config.add_route('favicon', '/favicon.ico')
    config.add_route('run_app', '/{app_url_name}', factory='tropofy.AppFactory')  # This needs to be at the end. It's a catch all!
    config.add_route('redirect_to_run_app', '/')  # This needs to be at the end. It's a catch all!


def groupfinder(email, request):
    '''Callback on authentication. Gets passed logged authenticated_userid (for us is email).
        Return groups that get permission (emails for now). Must match Factory.'''
    return [email]


class RootFactory(object):
    __acl__ = [
        (Allow, Everyone, 'view'),
        (Allow, Authenticated, 'logged_in'),
    ]

    def __init__(self, request):
        pass


class AppFactory(object):
    '''Factory for url's within apps (ajax calls). Needed so people can't innapropriately access ajax urls directly.'''
    def __init__(self, request):
        self.__acl__ = [(Allow, authenticated_userid(request), 'run_app')]