'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and governed by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import csv
from tropofy.file_io.class_tabular_data_mapping import ClassTabularDataMapping


class ClassTabularCsvMapping(ClassTabularDataMapping):
    """Maps a Python class to a tabular csv file.

    Used to both write to and read from csv with :func:`tropofy.file_io.read_write_xl.CsvReader` and :func:`tropofy.file_io.read_write_xl.CsvWriter`.

    :param class\_: Python class in mapping.
    :type class\_: class
    :param attribute_column_aliases: dict of class attribute to col alias.
    :type attribute_column_aliases: dict
    :param process_objects: (Optional) Function that processes the objects of ``class_`` loaded from Excel. Must accept a single parameter which will be passed a list of objects of class\_.
       If no function is passed, default behaviour is to assume ``class_`` is a SQLA class, and attempt to write the objects to the database.
    :type process_objects: function
    :param get_objects: (Optional) A function that accepts ``data_set`` as parameter and returns a list of objects of ``class_``
    :type get_objects: function
    :param objects: (Optional) A list of objects of ``class_`` to be included in export. If ``get_objects`` is supplied, these objects will be overridden.
    :type objects: list

    .. note:: To specify the order in which columns are written, use a collections.OrderedDict (Python built-in) for ``attribute_column_aliases``.
    """
    def __init__(self, class_, attribute_column_aliases, process_objects=None, get_objects=None, objects=None):
        super(ClassTabularCsvMapping, self).__init__(class_, attribute_column_aliases, process_objects, get_objects, objects)

class CsvReader(object):
    """Utility class for reading data from .csv files."""

    @classmethod
    def load_tabular_data_from_csv_file_in_memory(cls, data_set, file_in_memory, class_tabular_csv_mapping, delimiter=',', user_feedback_csv_file_identifier_str=''):
        """Loads data from a csv in memory file in a table format. Assumes first row is headings, and each subsequent row is data corresponding to the headings.

        :param data_set: Data set used to store the data read in.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param file_in_memory: File in memory. Common use case is to pass from :func:`tropofy.widget.file_upload_widget.process_file`
        :type file_in_memory: FileIO
        :param class_tabular_csv_mapping: Describes how the data and headings in the csv file map to a class.
        :type class_tabular_csv_mapping: :class:`tropofy.file_io.read_write_csv.ClassTabularCsvMapping`
        :param delimiter: (Optional) csv delimiter. Default ','
        :type delimiter: str
        :param user_feedback_csv_file_identifier_str: String to identify the csv file in error reporting.
        :ty[e user_feedback_csv_file_identifier_str: str
        """
        cls._load_tabular_data_from_csv_file(data_set, file_in_memory.file, class_tabular_csv_mapping, delimiter, user_feedback_csv_file_identifier_str)

    @classmethod
    def load_tabular_data_from_csv_file_on_disk(cls, data_set, file_path, class_tabular_csv_mapping, delimiter=','):
        """Loads data from a csv file in a table format. Assumes first row is headings, and each subsequent row is data corresponding to the headings.

        :param data_set: Data set used to store the data read in.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param file_path: Path on disk to the csv file.
        :type file_path: str
        :param class_tabular_csv_mapping: Describes how the data and headings in the csv file map to a class.
        :type class_tabular_csv_mapping: :class:`tropofy.file_io.read_write_csv.ClassTabularCsvMapping`
        :param delimiter: (Optional) csv delimiter. Default ','
        :type delimiter: str
        """
        cls._load_tabular_data_from_csv_file(data_set, open(file_path, 'rb'), class_tabular_csv_mapping, delimiter, file_path)

    @classmethod
    def _load_tabular_data_from_csv_file(cls, data_set, csv_file, class_tabular_csv_mapping, delimiter=',', user_feedback_csv_file_identifier_str=''):
        csv_dict_reader = csv.DictReader(csv_file, delimiter=delimiter)
        if class_tabular_csv_mapping.tabular_data_has_required_columns(csv_dict_reader.fieldnames, user_feedback_csv_file_identifier_str):
            objects = []
            for row in csv_dict_reader:
                attribute_values = dict((class_tabular_csv_mapping.get_attribute_from_column_name(col_heading), value) for col_heading, value in row.iteritems())
                objects.append(class_tabular_csv_mapping.class_(
                    **attribute_values
                ))
            class_tabular_csv_mapping.process_objects(data_set, objects)


class CsvWriter(object):
    @classmethod
    def write_tabular_data_to_csv_file_on_disk(cls, data_set, file_path, class_tabular_csv_mapping, delimiter=','):
        """Write data to a csv file in a table format. Sets first row as headings, and each subsequent row as data corresponding to the headings.

        If class_tabular_csv_mapping is not instantiated with parameters process_objects or objects, it will be assumed that the
        ``class_tabular_csv_mapping.class_`` is a SQLA ORM class. All objects in the database for this ``class_`` in the data_set will be written to the csv file.

        .. note:: Use a collections.OrderedDict in class_tabular_csv_mapping.attribute_column_alias to specify the output order.

        :param data_set: Data set used to store the data read in.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param file_path: Path on disk to the csv file.
        :type file_path: str
        :param class_tabular_csv_mapping: Describes how a class maps to headings and data in the csv file.
        :type class_tabular_csv_mapping: :class:`tropofy.file_io.read_write_csv.ClassTabularCsvMapping`
        :param delimiter: (Optional) csv delimiter. Default ','
        :type delimiter: str
        """
        csv_writer = csv.writer(open(file_path, 'w+'), delimiter=delimiter)

        # Write column names
        column_names = [k for k in class_tabular_csv_mapping.required_column_names]
        col_name_to_attribute_dict = class_tabular_csv_mapping.get_column_name_to_attribute_dict()
        csv_writer.writerow(column_names)

        objects = class_tabular_csv_mapping.objects if class_tabular_csv_mapping.objects is not None else class_tabular_csv_mapping.get_objects(data_set, class_tabular_csv_mapping.class_)
        if not objects:  # Default to getting SQLA objects
            objects = data_set.query(class_tabular_csv_mapping.class_)
        for object in objects:
            csv_writer.writerow([getattr(object, col_name_to_attribute_dict[col_name]) for col_name in column_names])
