"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

import importlib
import os
import inspect
import sys

from tropofy.app import LoadedApps, App, AppWithDataSets, AppDataSet, DataSetVar, SavedImageReference, CeleryTask, SynchronousTask
from tropofy.database.tropofy_orm import TropofyDbMetaData, DynamicSchemaName
from sqlalchemy.engine import reflection
from sqlalchemy.schema import CreateSchema
from sqlalchemy import create_engine


class AppManager(object):

    @staticmethod
    def get_app_from_url_name(url_name):
        return LoadedApps.apps.get(url_name)

    @staticmethod
    def get_app_if_only_one():
        if len(LoadedApps.apps) == 1:
            return AppManager.get_app_from_url_name(LoadedApps.apps.keys()[0])
        return None

    @classmethod
    def add_additional_absolute_paths_to_python_path(cls, additional_folders_str):
        """
        Additional folders to add to the python path enable including Python packages anywhere on the machine.
        The folder one up from an app location is always included by default (this is how the app is loaded).
        Use a ';' separated list for multiple paths.
        """
        if additional_folders_str:
            for dir_ in [dir_ for dir_ in additional_folders_str.split(";") if os.path.isdir(dir_)]:
                sys.path.append(dir_)

    @classmethod
    def initialise_apps(cls, apps_dir):
        if DynamicSchemaName.use_app_schemas:
            cls._create_schema_if_required(create_engine(App.overridden_global_sqla_engine), DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME)

        for dir_line in apps_dir.split("\n"):
            dirs = [dir_ for dir_ in dir_line.split(";") if os.path.isdir(dir_)]
            for dir_ in dirs:
                path, module_name = os.path.split(dir_)
                DynamicSchemaName.set_schema_name(module_name)
                sys.path.append(path)

                module = importlib.import_module(module_name)  # Importing the module here gives it the chance to set a custom value for DynamicSchemaName.schema_name

                AppManager._create_app_if_possible(
                    possible_app_package=module,
                    possible_app_schema_name=DynamicSchemaName.schema_name,
                    possible_app_package_file_path=dir_,
                )

    @classmethod
    def _create_app_if_possible(cls, possible_app_package, possible_app_schema_name, possible_app_package_file_path):
        """

        :param possible_app_package: A Python package that may contain an App (even though a package, is of type module)
        :type  possible_app_package: Python module
        :param possible_app_schema_name: The schema name to use for the app
        :type possible_app_schema_name: str
        :param possible_app_package_file_path: File path of possible app package.
        :type possible_app_package_file_path: str
        :return: tropofy.application.App
        """
        app = cls._get_app_interface_from_module(possible_app_package)
        if app:  # module_definition may not have a class that inherits from App.
            app.schema_name = possible_app_schema_name
            app.app_folder_path = possible_app_package_file_path
            app.init_db_engine()

            LoadedApps.apps.pop(app.url_name, None)
            LoadedApps.apps[app.url_name] = app
            cls._setup_app_db_tables(app)

    @classmethod
    def _get_app_interface_from_module(cls, imported_package):
        """
        :return: tropofy.application.App or None, if no App could be created from the package.
        """
        for class_name, class_type in inspect.getmembers(imported_package, inspect.isclass):
            if class_type not in [App, AppWithDataSets] and App in inspect.getmro(class_type):
                # Instantiate app
                return class_type(config=cls._get_config_from_module(imported_package))

    @classmethod
    def _get_config_from_module(cls, imported_package):
        for var_name, dict_ in inspect.getmembers(imported_package, lambda x: isinstance(x, dict)):
            if var_name == "CONFIG":
                return dict_

    @classmethod
    def _setup_app_db_tables(cls, app):
        if app.url_name != 'user_management':
            cls._create_schema_if_required(app.engine, app.schema_name)

            TropofyDbMetaData.metadata.create_all(
                bind=app.engine,
                tables=app.get_orm_tables() + [
                    TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(AppDataSet.__tablename__)],
                    TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(DataSetVar.__tablename__)],
                    TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(SavedImageReference.__tablename__)],
                    TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(CeleryTask.__tablename__)],
                    TropofyDbMetaData.metadata.tables[DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas(SynchronousTask.__tablename__)],
                ]
            )

    @classmethod
    def _create_schema_if_required(cls, engine, schema_name):
        """Schema creation is required if using app schemas, and the schema does not exist yet."""
        if schema_name and not cls._schema_exist(engine, schema_name):
            print("Creating schema %s..." % schema_name)
            engine.connect().execute(CreateSchema(schema_name))

    @classmethod
    def _schema_exist(cls, engine, schema_name):
        insp = reflection.Inspector.from_engine(engine)
        return schema_name in insp.get_schema_names()


class PossibleApp(object):
    def __init__(self, module, schema_name):
        self.module = module
        self.schema_name = schema_name