"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

import os
import sys
import inspect
import traceback
import collections
from pyramid.response import Response
from sqlalchemy import create_engine

from tropofy.file_io import read_write_xl
from data_set import AppDataSet
from tropofy.widgets import DataSets
from tropofy.database import DBSession
from tropofy.database.tropofy_orm import ORMBase, TropofyDbMetaData


class App(object):
    global_app_indexer = 0
    overridden_global_sqla_engine = ''

    def __init__(self, config=None):
        self.id = App.global_app_indexer
        App.global_app_indexer += 1
        self.config = config
        self.name = self.get_name()
        self.url_name = make_safe_url_name(self.name)
        self.schema_name = None
        self.developer_id = None
        self.step_groups = []
        self.app_folder_path = None  # Needs to be set dynamically after app created. This is to avoid params in constructor.
        self._init_step_groups()
        self.new_app_config_dict = {"has_none_template": True, "data_set_template_label": "Template"}
        self.new_app_config_dict.update(self.get_new_dataset_config())

    def init_db_engine(self):
        """
        Initialise database engine connection string. Is dynamic to enable for SQLite and PSQL connections
        Note: Must be called after self.app_folder_path is set.

        This has to be dynamic as otherwise it would require params to App.__init__ to be set. This would
        greatly complicate App definition for users.

        :raise AttributeError: self.app_folder_path is not set prior to function call.
        """
        if self.app_folder_path is None:
            raise AttributeError('self.app_folder_path must be set before App.init_db_engine is called.')

        if App.overridden_global_sqla_engine:
            self.engine = create_engine(App.overridden_global_sqla_engine)
        else:
            engine_path = os.path.join(self.app_folder_path, self.url_name + ".db")
            self.engine = create_engine("sqlite:///" + engine_path)


    def _init_step_groups(self):
        """Overide this to insert step groups before or after gui stepgroups"""
        self._add_gui_stepgroups_with_steps()

    def _add_gui_stepgroups_with_steps(self):
        [self.add_step_group(step_group) for step_group in self.get_gui()]

    def get_name(self):
        raise NotImplementedError

    def get_examples(self):
        return {}

    def get_new_dataset_config(self):
        """
        Return a dict of config options for the new dataset form
        keys:
            has_none_template- default True, type bool
            data_set_template_label- default "Template", type string
        :rtype: dict
        """
        return {}

    def get_example_app_names(self):
        return [k for k in self.get_examples()]

    def get_icon_url(self):
        return ''

    def get_gui(self):
        raise NotImplementedError

    def get_home_page_content(self):
        return {}

    def disable_forced_linear_workflow(self):
        return False

    def default_app_home_page_config(self):
        return {
            'content_app_name_header': '',
            'content_single_column_app_description': '',
            'content_double_column_app_description_1': '',
            'content_double_column_app_description_2': '',
            'content_row_2_col_1_header': '',
            'content_row_2_col_1_content': '',
            'content_row_2_col_2_header': '',
            'content_row_2_col_2_content': '',
            'content_row_2_col_3_header': '',
            'content_row_2_col_3_content': '',
            'content_row_3_col_1_header': '',
            'content_row_3_col_1_content': '',
            'content_row_3_col_2_header': '',
            'content_row_3_col_2_content': '',
            'content_row_4_col_1_header': '',
            'content_row_4_col_1_content': ''
        }

    def is_login_screen_default_create_account(self):
        """At the login screen the default can be create account or sign in.

        :rtype: bool
        """
        return True

    def get_published_license_key(self):
        return ''

    def get_home_page_configuration(self):
        config = self.get_home_page_content()
        ret_config = self.default_app_home_page_config().copy()
        ret_config.update(config)
        return ret_config

    def is_for_internal_tropofy_use(self):
        return self.is_for_internal_tropofy_use()

    def allows_preview_subscription(self):
        """Anyone can log in to an app and use it with example data if True"""
        return not self.is_for_internal_tropofy_use()  # Logic matches internal use for now. Corporate private apps etc will return False here.

    def add_step_group(self, step_group):
        self.step_groups.append(step_group)

    def get_ordered_step_groups(self):
        return sorted(self.step_groups, key=lambda step_group: step_group.id)

    def get_all_widgets(self):
        widgets = []
        for group in self.step_groups:
            for step in group.steps:
                widgets.extend(step.widgets)
        return widgets

    def get_dependency_ordered_dict_of_tables_mapped_to_classes_for_app(self):
        """
        :returns: ordered dict of sqlalchemy.sql.schema.Table: to SQL ORM Class
        """
        orm_classes = []
        for _, class_type in inspect.getmembers(sys.modules[self.__module__], inspect.isclass):
            if ORMBase in inspect.getmro(class_type):
                is_tropofy_framework_class = class_type.__module__.split('.')[0] == 'tropofy'
                if not is_tropofy_framework_class:
                    orm_classes.append(class_type)

        table_name_and_schema_to_orm_class = dict(
            ((TropofyDbMetaData.TableNameAndSchema(class_.__tablename__, self.schema_name), class_) for class_ in orm_classes)
        )
        dependency_sorted_tables = TropofyDbMetaData.get_dependency_sorted_tables(table_name_and_schema_to_orm_class.keys())

        dependency_ordered_dict_of_tables_mapped_to_classes = collections.OrderedDict()
        for table in dependency_sorted_tables:
            class_ = table_name_and_schema_to_orm_class[TropofyDbMetaData.TableNameAndSchema(table.name, self.schema_name)]
            dependency_ordered_dict_of_tables_mapped_to_classes[table] = class_

        return dependency_ordered_dict_of_tables_mapped_to_classes

    def get_orm_tables(self):
        return self.get_dependency_ordered_dict_of_tables_mapped_to_classes_for_app().keys()

    def get_data_set_as_excel_wb_string_repr(self, data_set):
        return read_write_xl.ExcelWriter.create_excel_string_repr(data_set)

    def export_data_set(self, data_set):
        response = Response(self.get_data_set_as_excel_wb_string_repr(data_set))
        response.headers['Content-Disposition'] = ("attachment; filename=export.xlsx")
        return response

    def get_path_of_file_in_app_folder(self, file_name):  # TODO: Should add docs to this. Is useful to be exposed.
        return os.path.join(self.app_folder_path, file_name)

    def get_parameters(self):  # Returns list of parameters
        return []

    def get_param(self, name):  # Returns parameter by name. Used by data_set to get parameter.
        return next(param for param in self.get_parameters() if param.name == name)


class AppWithDataSets(App):
    """Provides an interface for writing a Tropofy App.

    To create an App, a developer must implement the :class:`tropofy.app.AppWithDataSets` interface. This
    interface defines:

    * `App Name`
    * `User Interface`
    * `Example Data`

    The class below, :class:`MyFirstApp`, provides an example implementation of the :class:`AppWithDataSets`
    interface,

    .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
       :pyobject: MyFirstApp


    Refer to the documentation below for full details of the :class:`AppWithDataSets` interface.

    """

    def __init__(self, config=None):
        App.__init__(self, config)

    def _init_step_groups(self):
        self._add_data_sets_step_group()
        self._add_gui_stepgroups_with_steps()

    def _add_data_sets_step_group(self):
        step_group = StepGroup(name="Getting Started")
        step_group.add_step(Step(
            name='Select Data Set',
            hide_navigation=True,
            widgets=[DataSets()],
        ))
        self.add_step_group(step_group)

    def add_static_data_before_template_data_loaded(self, data_set):
        """Populate data set with static data, before template data loaded.

        :rtype: None
        """
        pass

    def create_new_data_set_for_user(self, user_email, example_data_set_name, specific_name=''):
        data_set = AppDataSet(self, user_email)
        if specific_name:
            data_set.name = specific_name
        else:
            data_set.name = example_data_set_name if example_data_set_name else "Empty"
        DBSession().add(data_set)
        DBSession().flush()

        data_set.init_parameters_with_defaults()  # Must be done here and not in AppDataSet.__init__ as requires data set to have ID.

        # TODO: load_master_data flag ?
        self.add_static_data_before_template_data_loaded(data_set)

        return data_set

    def populate_data_set_from_template(self, data_set, data_set_template_name):
        if data_set_template_name and self.get_examples():
            try:
                populate_example = self.get_examples().get(data_set_template_name)
                if populate_example:
                    populate_example(data_set)  # populates data set
            except Exception:
                print(traceback.format_exc())
                print("WARNING: An error occurred loading your example data set. See Traceback above.")


    def get_name(self):
        """This function returns the name of the App as a string.

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
           :pyobject: MyFirstApp.get_name

        :returns: the name of the App.
        :rtype: str

        """
        raise NotImplementedError

    def get_home_page_content(self):
        """(Optional) Define the contents of the apps default home page.

        :returns: Dictionary of {area_of_page:content_for_area} pairs
        :rtype: dict

        This function is used to define the contents for the default home page generated for an app.
        The default home page uses a fluid layout and includes a range of section into which content can be inserted.
        For sections that you don't want to use either leave them out of you dictionary or set their content to be an empty string.

        Example usage:

        .. literalinclude:: ../../../tropofy_example_apps/core_dependencies/facility_location/facility_location.py
           :language: python
           :pyobject: MyFacilityLocationSolverApp.get_home_page_content

        The set of all sections into which content can be inserted is defined by the keys in the default content
        dictionary which will be used if you do not implement this function:

        .. literalinclude:: ../../../tropofy/app/application.py
           :pyobject: App.default_app_home_page_config

        """
        return {}

    def get_examples(self):
        """Define example data sets that a user can load to demonstrate the Apps functionality.

        :returns: Dictionary of {example name:function} pairs, where each function populates an empty :class:`tropofy.app.AppDataSet`.
        :rtype: dict

        Example usage:

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_5.py
           :pyobject: MyFirstApp.get_examples


        An example of a function that populates a :class:`tropofy.app.AppDataSet` is given below,

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_5.py
           :pyobject: load_example_data

        .. note:: :func:`tropofy.app.AppWithDataSets.get_examples` returns a dictionary of strings mapped to functions, each function has the following signature:

           .. function:: example_data_function(data_set)

              :param data_set: an empty :class:`tropofy.app.AppDataSet`
              :type data_set: :class:`tropofy.app.AppDataSet`

        """
        return {}

    def get_default_example_data_set_name(self):
        """(Optional) Set the name of the example data set that is created by default on first use of the app.

        :returns: String that matches a key in the dictionary returned by :func:`tropofy.app.AppWithDataSets.get_examples`.
        :rtype: str

        If this function is not implemented or the string returned does not match an example, the first key
        alphabetically in the dictionary returned by :func:`tropofy.app.AppWithDataSets.get_examples` will be used.
        """
        return None

    def get_gui(self):
        """This function defines the user interface for an App.

        An App consists of a list of Step Groups, where each Step Group is made up of Steps. A Step Group is represented by the
        class :class:`tropofy.app.StepGroup`, and a Step is represented by the class :class:`tropofy.app.Step`.  An example user
        interface is provided below,

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
           :pyobject: MyFirstApp.get_gui


        :returns: list of step groups of type :class:`tropofy.app.StepGroup`, each consisting of steps of type :class:`tropofy.app.Step`.
        :rtype: list

        """
        raise NotImplementedError

    def get_parameters(self):
        """Define parameters to be used in the app.

        :returns: list of :class:`tropofy.app.Parameter` objects
        :rtype: list

        Example from :ref:`Plotting in 3D Example App<example_scipy_numpy_matplotlib_plotting_in_3d>`:

        .. literalinclude:: ../../../tropofy_example_apps/additional_dependencies/scipy_numpy_matplotlib/plotting_in_3d/plotting_in_3d.py
           :pyobject: MyApp.get_parameters

        The additional helper validating function used above is:

        .. literalinclude:: ../../../tropofy_example_apps/additional_dependencies/scipy_numpy_matplotlib/plotting_in_3d/plotting_in_3d.py
           :pyobject: validate_value_g_zero
        """
        return []

    def disable_forced_linear_workflow(self):
        """(Optional) Disable linear workflow, enabling users to move between steps non-linearly.

        :returns: Default of False
        :rtype: boolean
        """
        return False


def make_safe_url_name(name):
    return name.lower().replace(' ', '_').replace(',', '')  # Not entirely safe if people do crazy names!


class Parameter(object):
    '''A Parameter that can be accessed throughout the app.

    - Define in :func:`tropofy.app.AppWithDataSets.get_parameters`.
    - Expose simply with :class:`tropofy.widgets.ParameterForm`.
    - Get and set programatically with :func:`tropofy.app.AppDataSet.get_param` and :func:`tropofy.app.AppDataSet.set_param` respectively.

    :param name: Name of the parameter. Must be unique across all parameters and variables set with :func:`tropofy.app.AppDataSet.set_var`.
    :type name: str
    :param default: (optional) The default parameter value.
    :type default: Must comply with any restrictions imposed by the ``options``, ``validator``, and ``type`` parameters.
    :param options: (optional value restriction) An explicit list of values this parameter may take.
    :type options: list
    :param allowed_type: (optional value restriction) Force the value to be this type. Otherwise raise an exception.
    :type allowed_type: A Python Type. eg int, str, bool, dict, list
    :param validator: (optional value restriction) A function that provides custom validation of this parameter. Accepts a single param ``value`` and returns True for successful validation and a string explaining the error if not.
    :type validator: function
    :param label: (optional) Label to use on the parameter input form. name will be used if this is not supplied.
    :type label: str

    '''
    def __init__(self, name, default=None, options=None, allowed_type=None, validator=None, label=None):
        self.name = name
        self.default = default
        self.options = options
        self.allowed_type = allowed_type
        self.validator = validator
        self.label = label


class StepGroup(object):
    """A list of Step Groups forms the basis of an App.

    A Step Group consists of a list of Steps, where a Step is represented by a :class:`tropofy.app.Step` object.
    Adding a Step to a Step Group will append the Step to the end of the Step Group.  The example below
    shows how a :class:`StepGroup` is created,


    .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
       :pyobject: MyFirstApp.get_gui


    """
    step_group_indexer = 0

    def __init__(self, name="Step Group", steps=None):
        self.id = StepGroup.step_group_indexer
        StepGroup.step_group_indexer += 1
        self.name = name
        self.steps = steps if steps else []

    def add_step(self, step):
        """Adds a Step to the end of the Step Group.

          :param step: adds ``step`` to the end of the :class:`tropofy.app.StepGroup`.
          :type step: :class:`tropofy.app.Step`

        """
        self.steps.append(step)

    def get_ordered_steps(self):
        return sorted(self.steps, key=lambda step: step.id)

    def get_unique_div_id(self):  # TODO: Change to properties! (for step and widget also)
        return "step-group-" + str(self.id)

    def serialise(self):
        return {
            'name': self.name,
            'uniqueDivId': self.get_unique_div_id(),
            'steps': [step.serialise() for step in self.steps],
        }


class Step(object):
    """A list of Steps forms the basis of a Step Group.

    A Step consists of a list of Widgets, where a Widget is represented by a :class:`tropofy.app.Widget` object.
    Adding a Widget to a Step will append the Widget to the end of the Step.  The example below
    shows how a :class:`Step` is created,

    .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
       :pyobject: MyFirstApp.get_gui

    Using this technique, Widgets in the same Step will appear on different rows of the Step.  For a more fluid approach,
    i.e. where Widgets appear next to each other, a column span can be specified for each Widget.  The example below shows
    how this alternative approach can be used when a :class:`Step` is created,

    .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_5.py
       :pyobject: MyFirstApp.get_gui

    .. note :: The total number of columns for each Step is fixed at 12. When a row reaches capacity, additional Widgets
       will be forced onto a new row.

    Using this technique, Widgets can appear next to each other.  This is useful when creating dashboard like Apps.

    :param name: Name of the step
    :type name: str
    :param widgets: list of widgets, with possible layout dict as explained above.
    :type widgets: list
    :param help_text: (Optional) Text that will be displayed at the top of the step.
    :type help_text: str
    :param hide_navigation: (Optional) Hide the navigation accordion, next button, and back button when this step is visible. Default is False. Warning - you must implement another method of changing step if you set this to True, as otherwise users will become stuck.
    :type hide_navigation: bool
    """

    global_step_indexer = 0

    def __init__(self, name="Step", widgets=None, help_text=None, hide_navigation=False):
        self.id = Step.global_step_indexer
        Step.global_step_indexer += 1
        self.name = name
        self.help_text = help_text
        self.hide_navigation = hide_navigation
        self.widgets = []  # they get filled out in the func below
        self._construct_widgets(widgets)

    def _construct_widgets(self, widgets_in):
        """Fills out self.widgets from the list or dictionary passed.
        If dictionary, also fills out widget.num_cols_in_gui.

        Args: widgets. Same description as in __init__
        """
        for widget_def in widgets_in:
            widget = widget_def
            num_cols_in_gui = 12
            if type(widget_def) is dict:
                widget = widget_def["widget"]
                num_cols_in_gui = widget_def["cols"]
            self._construct_widget(widget, num_cols_in_gui)
            try:
                for embedded_widget in widget.embedded_widgets:
                    self._construct_widget(embedded_widget)
            except AttributeError:
                pass  # widget has no attribute embedded_widgets. Will be caught more often that not (in Python, exceptions are not exceptional!)

    def _construct_widget(self, widget, num_cols_in_gui=12):
        widget.allocate_id()
        widget.num_cols_in_gui = min(num_cols_in_gui, 12)
        self.widgets.append(widget)

    def get_jsonifiable_help_text(self):
        if self.help_text:
            text = self.help_text.replace('\n', '\\n')
            text = text.replace("'", "")
            return text

    def get_ordered_widgets(self):
        return sorted(self.widgets, key=lambda widget: widget.id)

    def get_unique_div_id(self):
        return "step-" + str(self.id)

    def serialise(self):
        return {
            'name': self.name,
            'uniqueDivId': self.get_unique_div_id(),
            'helpText': self.get_jsonifiable_help_text(),
            'hideNavigation': self.hide_navigation,
            'widgets': [widget.serialise() for widget in self.get_ordered_widgets()]
        }
