'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and governed by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from tropofy.database import DBSession
from tropofy.file_io.exceptions import TropofyFileImportExportException


class ClassTabularDataMapping(object):
    """"""
    def __init__(self, class_, attribute_column_aliases, process_objects=None, get_objects=None, objects=None):
        self.class_ = class_
        self.attribute_column_aliases = attribute_column_aliases
        self.process_objects = process_objects if process_objects is not None else ClassTabularDataMapping.process_sqla_objects
        self.get_objects = get_objects if get_objects is not None else ClassTabularDataMapping.get_all_sqla_objects_of_class_in_data_set
        self.objects = objects

    @property
    def class_name(self):
        return self.class_.__name__

    @property
    def required_column_names(self):
        """
        Returns all column names required in tabular data.

        :rtype: list
        """
        return self.attribute_column_aliases.values()

    def get_attribute_from_column_name(self, column_name):
        """
        :returns: Attribute as string that corresponds to column_name
        :rtype: string
        """
        for key, value in self.attribute_column_aliases.iteritems():
            if value == column_name:
                return key

    def get_column_name_to_attribute_dict(self):
        return dict(((v, k) for k, v in self.attribute_column_aliases.iteritems()))

    @staticmethod  # Use staticmethod as can't have cls as first attribute. Used as process_objects arg in ClassWsMapping.
    def process_sqla_objects(data_set, objects):
        data_set.add_all(objects)
        data_set.flush()

    @classmethod  # Use staticmethod as can't have cls as first attribute. Used as process_objects arg in ClassWsMapping.
    def get_all_sqla_objects_of_class_in_data_set(cls, data_set, class_):
        """Returns all mixins in the database of ``class_`` in ``data_set``.

        Used as default for parameter ``get_objects`` of :class:`tropofy.file_io.class_tabular_data_mapping.ClassTabularDataMapping`.
        """
        return data_set.query(class_).all()

    def tabular_data_has_required_columns(self, tabular_data_column_names_set, data_container_name):
        """
        :param tabular_data_column_names_set: Set of column names in tabular data
        :type tabular_data_column_names_set: set
        :param data_container_name: Name of data container. E.g, an Excel worksheet name, or a CSV file name.
        :type data_container_name: str
        """
        tabular_data_column_names_set = set(tabular_data_column_names_set)
        required_column_names_as_set = set(self.required_column_names)
        if required_column_names_as_set <= tabular_data_column_names_set:  # required_column_names_as_set is a subset of headers_as_set
            return True
        raise TropofyFileImportExportException('The first row in a sheet must be the column names. Sheet %s is missing the following columns %s' % (
            data_container_name, ", ".join(list(required_column_names_as_set - tabular_data_column_names_set))
        ))