'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from sqlalchemy.exc import IntegrityError
from sqlalchemy.orm import _mapper_registry
from sqlalchemy.schema import CheckConstraint

from tropofy.widgets import Widget, WidgetConfigException, Form
from tropofy.database.tropofy_orm import ValidationException
from tropofy.database import DBSession


class GridWidget(Widget):
    """
    Display data in a grid.

    :param title: A title for the grid.
    :type title: str
    :param desc_sort_cols: cols to sort descending
    :type desc_sort_cols: list of col names
    """
    ASCENDING = 'asc'
    DESCENDING = 'desc'
    CUSTOM_MAPPERS = {
        'format_date': 'formatDate',
        'format_time': 'formatTime',
        'format_datetime': 'formatDateTime',
    }

    def __init__(self, title=None, embedded_form_class=None, desc_sort_cols=None, widget_subscriptions=None):
        embedded_form_class = embedded_form_class if embedded_form_class else CreateUpdateForm
        create_update_form = embedded_form_class(parent_grid_widget=self)
        self.add_embedded_widget(name='createUpdateForm', widget=create_update_form)
        create_update_form.add_event_subscription('formSubmitSuccess', [self.actions('standingRefresh'), self.actions('hideCreateUpdateForm')])

        self.title = title
        self.desc_sort_cols = desc_sort_cols if desc_sort_cols else []

        super(GridWidget, self).__init__(widget_subscriptions=widget_subscriptions)

    def get_type(self):
        pass

    def refresh_from_db(self, data_set, request):
        col_names = self.get_grids_column_names()
        ordered_col_name_sort_direction_tuples = []
        for i in xrange(len(col_names)):
            try:
                ordered_col_name_sort_direction_tuples.append((
                    col_names[int(request.POST['order[%s][column]' % i])],
                    request.POST['order[%s][dir]' % i],
                ))
            except KeyError:
                break

        objects, total_records_before_filtering, total_records_after_filtering = self.get_filtered_refresh_objects(
            data_set=data_set,
            display_start=int(request.POST['start']),
            display_length=int(request.POST['length']),
            global_search_field=request.POST['search[value]'],
            ordered_col_name_sort_direction_tuples=ordered_col_name_sort_direction_tuples,
        )
        if objects == [] or objects:  # [] means no data in table = OK to proceed. None or something else falsy means an error and shouldn't proceed.
            return {
                'draw': int(request.POST['draw']),  # Suggested to cast to int to protect against cross-site-scripting attacks,
                'recordsTotal': total_records_before_filtering,
                'recordsFiltered': total_records_after_filtering,
                'data': [obj.as_json_data_row(self.get_grids_column_names()) for obj in objects]
            }

    def update_to_db(self, data, data_set_id, oper='', data_set=None):
        if oper == 'del':
            self.delete_row(obj_id=data['id'])
        return {'success': True}

    def serialise(self):
        """Join serialisation dict with Widget to get all params needed on client"""
        return dict(Widget.serialise(self).items() + {
            'columns': self.get_grids_column_names(),
            'hiddenCols': self.get_hidden_column_names(),
            'descSortedCols': self.get_decs_sorted_cols(),  # Cols default is to sort ascending on init. If in this list they will start sorted descending.
            'nonEditabledCols': self.get_non_editable_cols(),
            'gridIsEditable': self.grid_is_editable(),
            'title': self.get_title(),
            'colIndexToCustomFormatMaps': self._get_custom_formatting_mappers(),
            'datetimeDisplayFormats': self._get_datetime_display_formats(),
        }.items())

    def grid_is_editable(self):
        raise NotImplementedError

    def get_grids_column_names(self):
        raise NotImplementedError

    def get_non_editable_cols(self):
        """Will disable input boxes for these cols on edit in the GUI."""
        return []

    def get_decs_sorted_cols(self):  # TODO: 'desc' is typo - should be 'desc'
        return self.desc_sort_cols

    def get_hidden_column_names(self):
        return ['id', 'data_set_id']

    def get_refresh_objects(self, data_set):
        # Todo: Put source_class on GridWidget not SimpleGrid. SimpleGrid source_class must be SQLA class. GridWidget's does not (could be Python class or collections.NamedTuple.
        # Then, this function must return a set of source_class.
        # Do this and we can derive:
        # - GridWidget.get_column_form_input_types
        # - GridWidget.get_grids_column_names (must match source_class methods). Should probably add option to put labels on these so actual col names in grid change
        # - GridWidget.get_col_python_type (Doesn't exist yet - probably exists on FormWidget to determine saving add/edit string->python type conversion)

        # What to do about foreign keys? GridWidget shouldn't care! All on FormWidget. FormWidget select type get options & save options.
        raise NotImplementedError

    def get_filtered_refresh_objects(self, data_set, display_start, display_length, global_search_field, ordered_col_name_sort_direction_tuples):
        """
        :returns: (list of filtered mixins to display in grid for this page/search, total records in data set before filter, total records in data set after filter)
        :rtype: tuple - (<object list>, <int>, <int>)
        """
        raise NotImplementedError        

    def edit_row(self, data):
        raise NotImplementedError

    def add_new_row(self, data, data_set_id):
        raise NotImplementedError

    def delete_row(self, obj_id):
        raise NotImplementedError

    def get_table_hierachy_from_inheritance_chain(self):  # This is confusing. Shouldn't live on GridWidget - should be an implementation detail on a subclass.
        raise NotImplementedError

    def get_title(self):
        """Display a title for the grid. Set once for the grid - can not be dynamically changed.
        :rtype: str
        """
        return self.title

    def get_column_name_to_form_input_types(self):  # TODO for docs: Link to Form.Element.VALID_INPUT_TYPES
        """Input types to use in the form for cols. Must be in Form.Element.VALID_INPUT_TYPES. If not specified for column, will default to Form.Element.TEXT"""
        return {}

    def get_column_name_to_form_default(self):
        """Default values to use in the form corresponding to each column. Keys must correspond to column names.

        :return: dict of {column_name: default_value}
        """
        return {}

    def _get_datetime_display_formats(self):
        formats = {
            'date': "DD/MM/YYYY",
            'datetime': "HH:mm DD/MM/YYYY",
            'time': "HH:mm"
        }
        custom_formats = self.get_datetime_display_formats()
        if set(custom_formats.keys()) > Form.Element.VALID_DATETIME_INPUT_TYPES:
            raise WidgetConfigException("Error in function Form.get_datetime_display_formats. Must return dict with keys in: [Form.Element.DATE, Form.Element.TIME, Form.Element.DATETIME]")
        formats.update(custom_formats)
        return formats

    def get_datetime_display_formats(self):
        """Specify custom display formats to display datetimes, dates, and times in a grid and its edit form.

        Use format strings specified by `MomentJS <http://momentjs.com/docs/#/parsing/string-format/>`_. Note that these are slightly different to string-formats used in Python.

        Allowable keys are: 'date', 'datetime', and 'time'

        :returns: dict
        """
        return {}

    def _get_custom_formatting_mappers(self):
        col_names_to_form_input_types = self.get_column_name_to_form_input_types()
        col_names = self.get_grids_column_names()

        # Setup default mappers
        form_input_type_to_mapper = {
            'date': GridWidget.CUSTOM_MAPPERS['format_date'],
            'time': GridWidget.CUSTOM_MAPPERS['format_time'],
            'datetime': GridWidget.CUSTOM_MAPPERS['format_datetime'],
        }

        mappers = {name: form_input_type_to_mapper[col_names_to_form_input_types[name]] for name in col_names if col_names_to_form_input_types[name] in form_input_type_to_mapper.keys()}

        # Override with custom mappers
        custom_mappers = self.get_custom_formatting_mappers()
        if set(custom_mappers.keys()) > set(self.get_grids_column_names()):
            raise WidgetConfigException("Error in function Grid.get_custom_formatting_mappers. Must return dict with keys in columns names: %s" % col_names)

        mappers.update(custom_mappers)

        return {col_names.index(name): mapper for name, mapper in mappers.iteritems()}

    def get_custom_formatting_mappers(self):  # TODO: Write a custom mapper to put ',' in numbers and use as example.1
        """Customise the format of a particular column in the grid and edit form.

        Access mappers with ``GridWidget.CUSTOM_MAPPERS['mapper_name']

        Available Mappers:
         - 'format_date': Format date in display format specified in :func:`Grid.get_datetime_display_formats`
         - 'format_time': Format time in display format specified in :func:`Grid.get_datetime_display_formats`
         - 'format_datetime': Format datetime in display format specified in :func:`Grid.get_datetime_display_formats`

        :returns: dict of {col_name: mapper}
        """
        return {}


class CreateUpdateForm(Form):
    def __init__(self, parent_grid_widget):
        self.parent_grid_widget = parent_grid_widget

        super(CreateUpdateForm, self).__init__()

    def get_check_constraints_where_column_must_be_in_list_of_values(self, data_set):
        dict_cons = {}
        col_names = self._get_query_column_objects().keys()
        for con in self.parent_grid_widget.source_class.__table__.constraints:
            if type(con) is CheckConstraint and hasattr(con, 'sqltext') and hasattr(con.sqltext, 'text'):
                con_text = con.sqltext.text
                col_name = con_text.split()[0] if con_text.split() and con_text.split()[0] in col_names else None
                if col_name and con_text.find("in") != -1:
                    values_text = con_text[con_text.find("(")+1:con_text.find(")")]
                    dict_cons[col_name] = [v.replace("'", "") for v in values_text.split(",")]
        return dict_cons

    def get_options_for_column(self, data_set, col_name, hidden_col_names):
        options = None

        col_name_to_foreign_key_col_objects = {name: col for (name, col) in self._get_query_column_objects().items() if col.foreign_keys and col.name not in hidden_col_names}
        col_name_to_check_constraint_allowable_values = self.get_check_constraints_where_column_must_be_in_list_of_values(data_set)

        if col_name in col_name_to_foreign_key_col_objects.keys():
            column = col_name_to_foreign_key_col_objects[col_name]
            foreign_key_col = list(column.foreign_keys)[0].column
            mapper = _get_mapper_for_table(foreign_key_col.table)

            if data_set:
                options_qry = data_set.query(getattr(mapper.class_, foreign_key_col.name))
            else:
                options_qry = DBSession().query(getattr(mapper.class_, foreign_key_col.name))
            options = [Form.Element.Option(v[0]) for v in options_qry]

            if column.nullable:
                options = [Form.Element.Option(value=None)] + options

        elif col_name in col_name_to_check_constraint_allowable_values.keys() and col_name not in hidden_col_names:
            options = [Form.Element.Option(v) for v in col_name_to_check_constraint_allowable_values[col_name]]

        return options

    def get_form_elements(self, data_set):
        if self.parent_grid_widget.editable:  # TODO: Fix Jono's Hack to stop super large foreign key select drop downs loading in non editable grid. Nicer to remove whole form. Nicer again to maybe use typeahead with dynamic options... or something for FK's.
            form_elements = []
            non_editable_col_names = self.parent_grid_widget.get_non_editable_cols()
            hidden_col_names = self.parent_grid_widget.get_hidden_column_names()
            for name in self.parent_grid_widget.get_grids_column_names():
                input_type = self.parent_grid_widget.get_column_name_to_form_input_types().get(name, Form.Element.TEXT)
                default = self.parent_grid_widget.get_column_name_to_form_default().get(name, None)

                options = self.get_options_for_column(data_set, name, hidden_col_names)
                if options:
                    input_type = Form.Element.SELECT
                form_elements.append(Form.Element(
                    name=name,
                    input_type=input_type,
                    hidden=name in hidden_col_names,
                    disabled=name in non_editable_col_names,
                    options=options,
                    default=default,
                ))
            return form_elements
        return []

    def process_data(self, data, data_set):
        message = ''
        success = True
        for k, v in data.iteritems():  # Convert empty string entries to None. Needed to validate against is nullable on strings. (empty strings used on non nullable elsewhere in framework. Long term fix is to change this!)
            if v == '':
                data[k] = None
        if data['id']:  # Edit existing object
            try:
                self.parent_grid_widget.edit_row(data)
            except (ValidationException, IntegrityError) as e:
                message = e.message
                success = False
        else:  # Add new object
            try:
                self.parent_grid_widget.add_new_row(data, data_set.id if data_set else None)
            except (ValidationException, IntegrityError) as e:
                message = e.message
                success = False
        return {'success': success, 'message': message}

    def is_autosubmit(self, data_set):
        return False

    def _get_query_column_objects(self):  # TODO: This should be on simple grid as specific to SQLA.
        return {col.name: col for class_type in self.parent_grid_widget.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns}

    def get_datetime_display_formats(self):
        return self.parent_grid_widget._get_datetime_display_formats()


def _get_mapper_for_table(table):
    for mapper in list(_mapper_registry):
        if table in mapper._all_tables:
            return mapper
    return None