"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

from tropofy.database.db_manager import DbManager
from tropofy.widgets import GridWidget, Form
from sqlalchemy.sql.expression import desc, or_, cast
from sqlalchemy import String


class SimpleGrid(GridWidget):
    """Basic Grid. This widget is special. It is not used like the other widgets. You do not need to create a class
    which implements function to use an instance of this widget

    :param source_class: A user defined SQLAlchemy Python class
    :param editable: (Optional - default True). Enables grid to be edited.
    :type editable: bool
    :param widget_subscriptions: (Optional) Widget to subscribe to a triggered event of
    :type widget_subscriptions: list of tuples of (tropofy.widget, str of trigger_event_name, str of subscribed_action)

    .. note::
        The source_class provided to a SimpleGrid must have the following properties:
            - The database column names must be the same as the member variable names, i.e. you cannot use code like ``some_name = Column('some_other_name', String(50))``),
            - The initialisers parameter names must be the same as the member variable names. Note that SQLAlchemy provides a default initialiser with keyword arguments equal to the names of the column members defined for your class.

    The following class satisfies the above requirements

    .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
        :pyobject: ConformingClass
    """
    def __init__(self, source_class, editable=True, title=None, embedded_form_class=None, desc_sort_cols=None, widget_subscriptions=None):
        self.source_class = source_class
        self.editable = editable
        super(SimpleGrid, self).__init__(
            title=title,
            embedded_form_class=embedded_form_class,
            desc_sort_cols=desc_sort_cols,
            widget_subscriptions=widget_subscriptions
        )

    def get_type(self):
        return 'SimpleGrid'

    def add_new_row(self, data, data_set_id):
        edited_obj = DbManager.construct_object_and_add_to_db(source_class=self.source_class, data=data, data_set_id=data_set_id)
        return edited_obj.as_json_data_row(self.get_grids_column_names())

    def edit_row(self, data):
        edited_obj = DbManager.update_object_in_db(source_class=self.source_class, data=data)
        return edited_obj.as_json_data_row(self.get_grids_column_names())

    def delete_row(self, obj_id):
        DbManager.delete_object_from_db(source_class=self.source_class, obj_id=obj_id)

    def get_table_hierachy_from_inheritance_chain(self):
        full_class_hierachy = list(self.source_class.__bases__) + [self.source_class]
        return [class_type for class_type in full_class_hierachy if hasattr(class_type, "__table__")]

    def get_grids_column_names(self):
        return [col.name for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns]

    def get_column_name_to_form_input_types(self):
        return {col.name: Form._get_input_type_from_python_type(col.type.python_type) for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns}

    def get_filtered_refresh_objects(self, data_set, display_start, display_length, global_search_field, ordered_col_name_sort_direction_tuples):
        try:
            qry = self.get_qry_of_non_filtered_refresh_objects(data_set)
        except AttributeError:
            # TODO: Log that there was an attribute error on the refreshish
            return [], 0, 0
        total_records_before_filtering = qry.count()

        #Filter from search
        if global_search_field:
            search_conditions = []
            do_not_search_on_cols = self.get_col_names_to_not_search_on(data_set)
            for col_name in DbManager.sqla_cls_defined_column_names(self.source_class):
                if col_name not in do_not_search_on_cols:
                    search_conditions.append(
                        cast(getattr(self.source_class, col_name), String).ilike('%' + global_search_field + '%')  # .ilike is case insensitive .like
                    )
            qry = qry.filter(or_(*search_conditions))
                    
        total_records_after_filtering = qry.count()
        
        #Order cols
        for col_name, direction in ordered_col_name_sort_direction_tuples:
            if direction == GridWidget.ASCENDING:
                qry = qry.order_by(getattr(self.source_class, col_name))
            elif direction == GridWidget.DESCENDING:
                qry = qry.order_by(desc(getattr(self.source_class, col_name)))

        #Reduce to num rows on grid page
        qry = qry.offset(display_start).limit(display_length)

        return qry.all(), total_records_before_filtering, total_records_after_filtering

    def get_qry_of_non_filtered_refresh_objects(self, data_set):
        """
        :returns: SQLA qry object, that if executed, would return all mixins in the grid - unfiltered and unsorted.
        
        Assumes data_set exists.
        """
        if data_set and hasattr(self.source_class, 'data_set_id'):
            return data_set.query(self.source_class)
        raise AttributeError('SimpleGrid for source_class `%s` has no `data_set_id` column. SimpleGrid expects this. If not using data sets, extend SimpleGrid and overide `get_qry_of_non_filtered_refresh_objects`' % self.source_class.__name__)

    def get_col_names_to_not_search_on(self, data_set):
        return self.get_hidden_column_names()

    def grid_is_editable(self):
        return self.editable
