"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

from datetime import datetime as dt
import time
import os
import shutil
import json
from datetime import datetime
from pyramid import threadlocal
from sqlalchemy.types import Text, Integer, DateTime, Boolean, PickleType
from sqlalchemy.schema import Column, UniqueConstraint
from sqlalchemy.orm import relationship
from sqlalchemy import event
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.sql.expression import desc
from tropofy.database.tropofy_orm import ORMBase, DataSetMixin, ExtendedJsonEncoder, Validator, ValidationException
from tropofy.database import DBSession
from tropofy.app import LoadedApps
from tropofy.database.tropofy_orm import DynamicSchemaName
from celery import current_task
from celery.result import AsyncResult
import re


class TaskMessage(object):
    MUTED = "muted"
    PRIMARY = "primary"
    SUCCESS = "success"
    INFO = "info"
    WARNING = "warning"
    ERROR = "danger"

    def __init__(self, text, status=None):
        self.text = text
        self.status = status

    def to_html(self):
        return "<li class='text-%s'>%s</li>" % (self.status, self.text)


class AppDataSet(ORMBase):
    """A Data Set represents a collection of information relating to an App and owned by a User, a User
    could have many Data Sets all relating to one App.

    The :class:`AppDataSet` provides an interface for interacting with the Data Set.  The example below
    shows how a :class:`AppDataSet` is used,

    .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_5.py
       :pyobject: load_example_data

    In addition, :class:`AppDataSet` exposes the SQLAlchemy :class:`Query` class.  This provides a
    mechanism for easily searching a Data Set.  The example below shows how a Data Set can be queried,

    .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
       :pyobject: MyKMLMap.get_kml

    """

    type_str = Column('type', Text)
    app_name = Column(Text, nullable=False)
    user_email = Column(Text, nullable=False)
    name = Column(Text)
    created_datetime = Column(DateTime)
    last_accessed_datetime = Column(DateTime)
    current_step_index = Column(Integer, nullable=False)
    latest_enabled_step_index = Column(Integer, nullable=False)
    execute_functions_info = Column(Text)
    saved_image_references = relationship('SavedImageReference')
    
    __table_args__ = (
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {},
    )

    db_session = DBSession

    def __init__(self, app, user_email, name=''):
        self.app_name = app.name
        self.user_email = user_email
        self.name = name if name != '' else 'New data set'
        self.created_datetime = dt.now()
        self.current_step_index = 1  # Default step to show after data_set selected. 1 not 0 as data_set_selection is step 0.
        self.latest_enabled_step_index = 1
        self.execute_functions_info = ''

    def send_progress_message(self, message, status=None):
        """Sends message(s) to the user about the progress of some operation.

        :param status: Status of message(s) - use available statuses in :func:`tropofy.app.AppDataSet.message_status`. Colours the message.
        :type status: str

        :param message: Adds ``message`` to the current progress message for the data set.
        :type message: string, or list of strings. If list, each element is a separate message written on a new line.
        """
        msg_strings = message if not isinstance(message, basestring) else [message]
        new_messages = []
        for msg in msg_strings:
            if msg in ["<br>", "\r\n", "\n"]:
                new_messages.append(TaskMessage(text="<br>", status=status))
            for msg_split_new_lines in re.split("<br>|\n", msg):  # A lot of code (often third party), adds newlines with <br> or \n explicitly into them. This will strip these and make new lines.
                new_messages.append(TaskMessage(text=msg_split_new_lines, status=status))

        try:  # Using Celery
            current_task._running_task_meta.add_messages(new_messages)
            current_task.update_state(
                state='PROGRESS',
                meta=current_task._running_task_meta
            )
        except:  # Not using Celery current_task doesn't exist.
            try:
                self._running_task.add_messages(new_messages)
            except AttributeError:  # no attr self._running_task
                print("Couldn't send progress messages: {}".format([m.text for m in new_messages]))  # Called outside execute function widget?

    def current_celery_task_terminated(self):
        return self.query(CeleryTask.terminated_by_user).filter_by(task_id=current_task.request.id).scalar()

    def communicate_with_subprocess(self, subprocess, fail_task_if_piped_error_message=True, subprocess_error_msg=None):
        """Sends progress messages from subprocess. Also terminates the task and subprocess if user has terminated through GUI (Celery only).

        .. warning:: To communicate with the subprocess, ``subprocess`` must have been initialised with param ``stdout=subprocess.PIPE``. Furthermore, to report errors in the subprocess, pass the additional parameter to subprocess initialiser ``stderr=subprocess.PIPE``. By default, it is assumed that if a subprocess pipes any text to ``stderr``, that the task has failed. Set parameter ``fail_task_if_piped_error_message`` to ``False`` to change this.

        .. note:: Some subprocesses use stderr for additional logging. In this circumstance, it may be useful to redirect ``stderr`` output to ``stdout``. To do this, use ``stderr=subprocess.STDOUT`` when initialising ``subprocess``. This will treat output to ``stderr`` as if it were ``stdout`` (i.e. cannot stop the app even if param ``fail_task_if_piped_error_message`` is ``True``).

        **Example:** Calling a subprocess and communicating with it. It is assumed that ``subprocess`` has been imported and that this code is within :func:`tropofy.widgets.ExecuteFunctionWidget.execute_function`, which defines ``data_set`` as an instance of :class:`tropofy.app.AppDataSet`.

        .. code-block:: python

            p = subprocess.Popen(["process_details"], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            data_set.communicate_with_subparocess(p)

        :type subprocess: subprocess.Popen
        :param fail_task_if_piped_error_message: (Default ``True``) Determines if task should be marked as failed (won't commit db changes) if there are piped error messages with ``stderr``.
        :type fail_task_if_piped_error_message: bool
        :param subprocess_error_msg: (Default `"Error during subprocess execution:"`) Message to display before writing error messages.
        :type subprocess_error_msg: str
        """
        lines = []
        last_update_time = time.time()
        while subprocess.poll() is None:
            if current_task:  # Using Celery
                if self.current_celery_task_terminated():  # TODO: Move into every 2 second loop?
                    subprocess.kill()
                    raise Exception()

                try:
                    lines.append(subprocess.stdout.readline()) # This is blocking see http://stackoverflow.com/questions/375427/non-blocking-read-on-a-subprocess-pipe-in-python
                    if time.time() - last_update_time > 2:  # send_progress_message includes a db hit so is a bit slow. Sending less than every 2 seconds is useless, as this is polling frequency.
                        self.send_progress_message(lines, status=self.message_status.MUTED)
                        lines = []
                        last_update_time = time.time()
                except AttributeError:
                    pass  # subprocess.stdout not set for subprocess.

        try:
            lines += subprocess.stdout.readlines()  # Add remaining lines
        except AttributeError:
            pass  # subprocess.stdout not set for subprocess.

        self.send_progress_message(lines, status=self.message_status.MUTED)

        errors = subprocess.stderr.readlines() if subprocess.stderr else None
        if errors:
            subprocess_error_msg = subprocess_error_msg if subprocess_error_msg else "Error during subprocess execution:"
            self.send_progress_message([subprocess_error_msg], status=self.message_status.ERROR)
            self.send_progress_message(errors, status=self.message_status.ERROR)

            if fail_task_if_piped_error_message:
                raise Exception()

    class MessageStatusEnum(object):
        MUTED = TaskMessage.MUTED
        PRIMARY = TaskMessage.PRIMARY
        SUCCESS = TaskMessage.SUCCESS
        INFO = TaskMessage.INFO
        WARNING = TaskMessage.WARNING
        ERROR = TaskMessage.ERROR

    @property
    def message_status(self):
        return AppDataSet.MessageStatusEnum

    @property
    def created_datetime_str(self):
        return self.created_datetime.strftime('%Y-%m-%d - %H:%M%p') if self.created_datetime else ''

    @property
    def last_accessed_datetime_str(self):
        return self.last_accessed_datetime.strftime('%Y-%m-%d - %H:%M%p') if self.last_accessed_datetime else ''

    def update_time_accessed(self):
        self.last_accessed_datetime = dt.now()

    @property
    def app_module_path(self):  # Not added as a col to keep database independent of app folder structure.
        try:
            return self.app.app_folder_path
        except Exception as e:  # TODO: Which exception? Narrow it down...
            try:
                return self._app_module_path
            except AttributeError:
                raise Exception("AppDataSet.app_module_path cannot be referenced. AppDataSet._app not accessible, thus expects this is a Celery Task with _app_module_path set.")

    @property
    def app_url_name(self):
        try:
            return self.app.url_name
        except Exception as e:  # TODO: Which exception? Narrow it down...
            try:
                return self._app_url_name
            except AttributeError:
                raise Exception("AppDataSet.app_url_name cannot be referenced. AppDataSet._app not accessible, thus expects this is a Celery Task with _app_url_name set.")

    @property
    def app(self):
        for app in LoadedApps.apps.values():
            if app.name == self.app_name:
                return app
        raise AttributeError("""
            Cannot reference AppDataSet.app here. Is this code being run by a Celery worker?
            Celery workers cannot reference data_set.app. Using data_set.app_module_path and data_set.app_url_name may get around this for you.
        """)

    @property
    def file_save_folder(self):
        """Returns a path on disk of a folder specific to the data set to save files to.

        Additionally, creates the folder for the data set if it does not yet exist.
        """
        folder = os.path.join(self.app_module_path, 'data_set_data', 'data_set_' + str(self.id))
        if not os.path.exists(folder):
            os.makedirs(folder)
        return folder

    def add(self, obj):
        """Adds a TropofyBase object to the Data Set.

        :param obj: Adds ``obj`` to the Data Set.
        :type obj: :class:`tropofy.app.TropofyBase()`

        """
        obj.data_set_id = self.id
        session = self.db_session()
        session.add(obj)
        session.flush()
        return obj

    def add_all(self, objects):
        """Adds a list of TropofyBase mixins to the Data Set.  The example below
        shows how :func:`add_all` is used,

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_5.py
           :pyobject: load_example_data

        :param objects: Adds ``obj`` to the Data Set.
        :type objects: list

        """
        for obj in objects:
            obj.data_set_id = self.id

        session = self.db_session()
        session.add_all(objects)
        session.flush()

    def flush(self):
        self.db_session().flush()

    def query(self, *args):
        """Returns an SQLAlchemy :class:`Query` object for the Data Set.  The example below
        shows how :func:`query` is used,

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
           :pyobject: MyKMLMap.get_kml

        :param args: Classes to query, replicates SQLA functionality.
        :type args: Variable number of arguments. Each should be of type :class:`tropofy.app.TropofyBase()` or a Column on such a class. See the `SQLA query docs <http://docs.sqlalchemy.org/en/rel_0_9/orm/query.html#the-query-object>`_ for help.

        :returns: Returns a ``Query`` object populated by with all mixins of type ``source_class``
        """
        return self.db_session().query(*args).filter_by(data_set_id=self.id)

    def save_image(self, name, image):
        """Saves a file to disk, and manages its lifetime and location..

        :param name: Name of file. Must include file type extension such at .png or .jpg.
        :type name: string
        :param image: File stream
        :type image: Image
        :rtype: None
        """

        img_reference = self._get_image_reference_by_name(name)
        if img_reference:
            img_reference.save_image(image)
        else:
            img_reference = SavedImageReference(image_name=name)
            self.add(img_reference)
            img_reference.save_image(image)

    def get_image_path(self, name):
        """Returns the path to serve statically of a file saved with :func:`tropofy.app.AppDataSet.save_image`

        This function is usually used with :class:`tropofy.app.StaticImage`.

        :param name: Name of file (must match the name of an image saved with :func:`tropofy.app.AppDataSet.save_image`)
        :type name: string
        :rtype: string

        For example, for a file called 'output.png' saved with :func:`tropofy.app.AppDataSet.save_image`:

        .. code-block:: python

           data_set.get_image_path('output.png')
        """

        ref = self._get_image_reference_by_name(name)
        return ref.static_path if ref else None

    def _get_image_reference_by_name(self, name):
        return next((ref for ref in self.saved_image_references if ref.image_name == name), None)

    def get_path_of_file_in_data_set_folder(self, file_name):
        """Returns a path on disk, within a folder specific for the data set, that is safe to save the file to."""
        return os.path.join(self.file_save_folder, file_name)

    def get_var(self, name):
        """Get a variable that was set with :func:`tropofy.app.AppDataSet.set_var`"""
        try:
            return self.query(DataSetVar).filter_by(name=name).first().value
        except AttributeError:  # param doesn't exit
            return None

    def set_var(self, name, value):
        """Set a variable that can be accessed with :func:`tropofy.app.AppDataSet.get_var`"""
        var = self.query(DataSetVar).filter_by(name=name).first()
        if var:
            var.value = value
        else:
            self.add(DataSetVar(name=name, value=value)) 

    def get_param(self, name):
        """Return the value of a :class:`tropofy.app.Parameter`

        :param name: Name of parameter. Must match a parameter name defined in :func:`tropofy.app.AppWithDataSets.get_parameters`
        :type name: str

        :returns: Parameter value
        """
        return self.get_var(name)

    def set_param(self, name, value):
        """Set the value of a :class:`tropofy.app.Parameter`

        :param name: Name of parameter. Must match a parameter name defined in :func:`tropofy.app.AppWithDataSets.get_parameters`
        :type name: str
        :param value: Value of parameter. Must satisfy any value restrictions specified on the parameter.
        """
        param = self.app.get_param(name)
        if param:
            if param.allowed_type:  # Will check correct type. Will also convert to correct type if able.
                value = Validator.validate(value, param.allowed_type, name)
            if param.validator:
                validation = param.validator(value)
                if validation != True:
                    raise ValidationException(name=name, value=value, message=validation)
            self.set_var(name, value)
        else:
            raise ValidationException(name=name, value=value, message='Parameter does not exist.')

    def init_parameters_with_defaults(self):  # Init parameter defaults on data set creation
        for param in self.app.get_parameters():
            self.set_param(name=param.name, value=param.default)

    def get_tasks_qry(self, widget_unique_name, get_only_celery_tasks, limit=None):
        """Return a list of tasks, sorted by most recent first.
        :param widget_unique_name: Get only tasks for a certain widget
        :type widget_unique_name: str
        :param get_only_celery_tasks: Return only tasks executed with Celery. To return both Celery and non Celery would task much longer (couldn't lean on db much), and probably has no use case.
        :type get_only_celery_tasks: bool
        :param limit: Limit the number of tasks returned.
        :type limit: int
        """
        class_ = CeleryTask if get_only_celery_tasks else SynchronousTask
        tasks_qry = self.query(class_).filter_by(widget_unique_name=widget_unique_name).order_by(desc(class_.start_datetime))
        if limit is not None:
            tasks_qry = tasks_qry.limit(limit)
        return tasks_qry

    def get_task(self, task_id):
        """Return a task referenced by the ID. Checks both CeleryTask and SynchronousTask"""
        try:
            return self.query(CeleryTask).filter_by(task_id=task_id).one()
        except NoResultFound:
            try:
                return self.query(SynchronousTask).filter_by(id=task_id).one()  # Uses id for task_id
            except NoResultFound:  # Task not found
                assert False  # TODO: return something saying the task id is wrong.

    __mapper_args__ = {   # Todo: Remove these? No inheritance appears to be happening so not needed...
        'polymorphic_on': type_str,
        'polymorphic_identity': 'AppDataSet'
    }


class DataSetVar(DataSetMixin):
    name = Column(Text, nullable=False)
    value_as_json_str = Column(Text, nullable=True)
    stored_type = Column(Text)  # Stores the type of object that got stored, so that it can be decoded back out. 

    __table_args__ = (
        UniqueConstraint('name', 'data_set_id'),
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {}
    )

    def __init__(self, name, value=None, type=None):
        self.name = name
        self.value = value
        
    @property
    def value(self):        
        json_str = json.loads(self.value_as_json_str)
        return Validator.validate(json_str, self.stored_type, self.name)

    @value.setter
    def value(self, value):
        try:
            self.value_as_json_str = json.dumps(value, cls=ExtendedJsonEncoder)
            self.stored_type = type(value).__name__
        except TypeError:
            raise


class TaskInterfaceMixin(object):
    QUEUED = "QUEUED"
    FAILURE = "FAILURE"
    SUCCESS = "SUCCESS"
    RUNNING = "RUNNING"
    TERMINATED = "TERMINATED"

    def __init__(self, widget_unique_name):
        self.widget_unique_name = widget_unique_name
        self.start_datetime = datetime.now()

    @declared_attr
    def start_datetime(cls):
        return Column(DateTime)

    @declared_attr
    def widget_unique_name(cls):
        return Column(Text, nullable=False)

    @property
    def task_id(self):
        raise NotImplementedError

    @property
    def status(self):
        raise NotImplementedError

    @property
    def end_datetime(self):
        raise NotImplementedError

    @property
    def messages(self):
        raise NotImplementedError

    @property
    def is_currently_running(self):
        raise NotImplementedError

    @property
    def duration(self):
        if self.end_datetime:
            return self.end_datetime - self.start_datetime
        else:
            return datetime.now() - self.start_datetime

    @property
    def elapsed_seconds(self):
        return round(self.duration.total_seconds())

    @property
    def status(self):
        raise NotImplementedError

    def task_can_be_terminated(self):
        return False

    def terminate(self):
        raise NotImplementedError

    def serialise(self):
        return {
            'taskId': self.task_id,
            'dateTimeStartedStr': self.start_datetime,
            'elapsedSeconds': self.elapsed_seconds,
            'progressMessagesHtml': "".join([m.to_html() for m in self.messages[-2000:]]),
            'isCurrentlyRunning': self.is_currently_running,
            'status': self.status,
            'taskCanBeTerminated': self.task_can_be_terminated(),
            #'summaryMessage': execute_fn_info.last_run_messages
        }


class CeleryTask(DataSetMixin, TaskInterfaceMixin):
    QUEUED = 'PENDING'  # SD Note: 'PENDING' will be returned by Celery even if the task_id not known. Be careful of this - hopefully never an issue.
    STARTED = 'STARTED'
    PROGRESS = 'PROGRESS'
    SUCCESS = 'SUCCESS'
    FAILURE = 'FAILURE'
    TERMINATED = 'REVOKED'
    TASK_COMPLETE_STATUS = [SUCCESS, FAILURE, TERMINATED]

    task_id = Column(Text, nullable=False)
    terminated_by_user = Column(Boolean, nullable=True)
    _finished_result = Column(PickleType, nullable=True)

    __table_args__ = (
        UniqueConstraint('data_set_id', 'task_id'),
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {}
    )

    def __init__(self, task_id, widget_unique_name):
        self.task_id = task_id
        self.terminated_by_user = False
        TaskInterfaceMixin.__init__(self, widget_unique_name)

    @property
    def _async_result(self):
        return AsyncResult(self.task_id)

    @property
    def _result(self):
        result = self._finished_result if self._finished_result is not None else self._async_result.result
        if type(result) is TaskResultMeta:  # Stop returning non dict results. E.g. if an exception, self._async_result.result will equal the exception.
            return result

    @property
    def messages(self):
        try:
            return self._result.messages
        except AttributeError:
            return []

    @property
    def is_currently_running(self):  # Is currently executing OR QUEUED.  # TODO: Make naming better here. Goes all the way to the GUI
        return self._is_currently_executing_on_worker or self._is_currently_queued

    @property
    def _is_currently_executing_on_worker(self):
        return not (self._is_currently_queued or self._is_finished_or_terminated)

    @property
    def _is_currently_queued(self):
        if self._is_finished_or_terminated:
            return False
        try:
            return self._async_result.status == CeleryTask.QUEUED
        except AttributeError:  # self._async_result doesn't have a status
            return False

    @property
    def _is_finished_or_terminated(self):
        return bool(self._finished_result)

    @property
    def end_datetime(self):
        try:
            return self._result.end_datetime
        except AttributeError:
            pass  # self._result is None. (Warning! Check self._result and then returning self._result.end_datetime can cause an error as self._result is changed in another process)

    @property
    def status(self):
        if self._is_currently_queued:
            return TaskInterfaceMixin.QUEUED
        elif self._is_currently_executing_on_worker:
            return TaskInterfaceMixin.RUNNING

        finished_status = self._finished_result.finished_status
        if finished_status == CeleryTask.SUCCESS:
            return TaskInterfaceMixin.SUCCESS
        elif finished_status == CeleryTask.TERMINATED:
            return TaskInterfaceMixin.TERMINATED
        else:  # finished_status == CeleryTask.FAILURE:
            return TaskInterfaceMixin.FAILURE

    @property
    def is_finished_and_successful(self):
        """
        :rtype: bool
        """
        return self.status == CeleryTask.SUCCESS

    def terminate(self):
        """
        Terminate a running task. This will capture the latest task result (msgs etc.) and store them as a finished result.

        ..note:: This will not actually stop a running process! Need to use rabitmq as celery msg broker for this to work - we are using AWS SQS which doesn't support 'revoke'.
         It is however possible within app code to check :func:`tropofy.app.AppDataSet.current_celery_task_terminated` and raise an exception (cancelling the task) if required.
         If a subprocess is running, use p.kill() to kill it. A good place to check if the task has been terminated, is in the while loop that is reading msgs from a subprocess.
        """
        self.terminated_by_user = True
        task_meta = TaskResultMeta() if self._result is None else self._result

        task_meta.add_terminated_message()
        task_meta.finalise_task(finished_status=CeleryTask.TERMINATED)
        self._finished_result = task_meta

    def task_can_be_terminated(self):
        return True


class SynchronousTask(DataSetMixin, TaskInterfaceMixin):

    _is_currently_running = Column(Boolean, nullable=False)
    _messages = Column(PickleType, nullable=False)  # List of TaskMessage
    _status = Column(Text, nullable=False)
    _finished_result = Column(PickleType, nullable=True)
    end_datetime = Column(DateTime)

    __table_args__ = (  # Don't need extra unique constraint on task_id, as just using self.id as task_id.
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {}
    )

    def __init__(self, widget_unique_name):
        self._is_currently_running = True
        self._status = TaskInterfaceMixin.RUNNING
        self._messages = []
        TaskInterfaceMixin.__init__(self, widget_unique_name)

    @property
    def task_id(self):
        return self.id

    @property
    def messages(self):
        return self._messages

    @property
    def is_currently_running(self):
        return self._is_currently_running

    def track_task_finished(self, execution_success, exception=None):
        if not execution_success and str(exception):
            self.add_messages([TaskMessage(text="%s: %s" % (type(exception).__name__, str(exception)), status=TaskMessage.ERROR)])
        self.end_datetime = datetime.now()
        self._status = TaskInterfaceMixin.SUCCESS if execution_success else TaskInterfaceMixin.FAILURE
        self._is_currently_running = False

    def add_messages(self, messages):
        self._messages = self._messages + messages

    @property
    def status(self):
        return self._status

    # def terminate(self):  # THIS DOESNT WORK! SEAMS TO LOCK THE SESSION. REALLY STUPID - JUST LEAVING IT FOR NOW.
    #     """Note: This is no way actually terminates the task execution. Just flags the task as ended. The server will keep running the task as there is no way to tell it to stop without Celery"""
    #     if self._is_currently_running:  # Can't terminate an already terminated task.
    #         self.end_datetime = datetime.now()
    #         self._is_currently_running = False
    #         self._status = TaskInterfaceMixin.TERMINATED


class TaskResultMeta(object):
    """Stores metadata about a celery task. Attached to current_task to store metadata while the task is running. Stored in CeleryTask._finished_task_meta for finished task."""

    def __init__(self):
        self.messages = []
        self.end_datetime = None
        self.progress = None
        self.title = None
        self._finished_status = None

    @property
    def finished_status(self):
        return self._finished_status

    @finished_status.setter
    def finished_status(self, status):
        if status in CeleryTask.TASK_COMPLETE_STATUS:
            self._finished_status = status
        else:
            raise Exception("Celery task status Exception: '%s' is not a valid finished Celery task status." % status)

    def add_messages(self, messages):
        """
        :type messages: list of :class:`tropofy.app.data_set.TaskMessage`
        """
        self.messages += messages

    def add_exception_message(self, exception):
        """
        :type exception: Exception
        """
        self.messages.append(TaskMessage(text="%s: %s" % (type(exception).__name__, str(exception)), status=TaskMessage.ERROR))

    def add_terminated_message(self):
        self.messages.append(TaskMessage(text="Task Stopped.", status=TaskMessage.ERROR))

    def finalise_task(self, finished_status):
        """Finalise a task. Add appropriate final messages

        Writing task rollback msg here assumes :func:`tropofy.celery.task_postrun_handler` is not committing db for unsuccessful tasks (error and terminated).
        Needs to be here so that the msg is displayed immediately after termination, even though a terminated task may continue to run on
        the server for a time, and only be rolled back and not committed once finished.
        """
        self.end_datetime = datetime.now()
        self.finished_status = finished_status
        if self.finished_status != CeleryTask.SUCCESS:
            self.messages.append(TaskMessage(text="Task execution did not complete successfully - database changes have not been committed.", status=TaskMessage.ERROR))


class SavedImageReference(DataSetMixin):
    image_name = Column(Text, nullable=False)
    file_path = Column(Text)
    datetime_saved = Column(DateTime)
    data_set = relationship('AppDataSet')

    __table_args__ = (
        UniqueConstraint('image_name', 'data_set_id'),
        {"schema": DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME} if DynamicSchemaName.use_app_schemas else {}
    )

    def __init__(self, image_name):
        self.image_name = image_name

    def save_image(self, image):
        folder = self.data_set.file_save_folder
        self.delete_referenced_file()  # Delete if it exists
        if not os.path.exists(folder):
            os.makedirs(folder)

        self.datetime_saved = dt.now()
        self.file_path = os.path.join(
            folder,
            str(int(time.mktime(self.datetime_saved.timetuple()))) + self.image_name
        )

        image.save(self.file_path)

    @property
    def static_path(self):
        request = threadlocal.get_current_request()
        app = request.app  # static_path is of folder apps_static/app_url_name/data_set_x/filename
        return os.path.join(
            'apps_static',
            app.url_name,
            os.path.relpath(self.file_path, request.app.app_folder_path)
        )

    def delete_referenced_file(self):
        try:  # Remove old file if it exists
            os.remove(self.file_path)
        except (TypeError, OSError):
            pass


@event.listens_for(AppDataSet, 'before_delete')
def receive_before_delete(mapper, connection, data_set):
    try:
        shutil.rmtree(data_set.file_save_folder)
    except OSError:
        pass
