import os
import xml.etree.cElementTree as ET
from xml.dom import minidom


class XmlInterfaceMixin(object):
    """Inherit from this mixin to be able to use objects of the class as items in :func:`tropofy.file_io.read_write_xml.XmlWriter.write_xml_file`"""
    def xml_data(self, root):
        return self.__dict__


class XmlWriter(object):
    """Utility class for writing data to XML files."""
    @classmethod
    def write_xml_file(cls, items, file_name, folder_for_files):
        """
        :param items: list of objects of classes that inherit from :class:`tropofy.file_io.read_write_xml.XmlInterfaceMixin`
        :param file_name: name of the xml file, including '.xml'. Note that this does not include the path
        :type file_name: str
        :param folder_for_files: absolute file path to save xml files to - (Tip: use ``data_set.file_save_folder`` to get a unique folder for each data_set).
        :type folder_for_files: str
        """
        root = ET.Element("DataInfo")
        cls._write_xml_nodes(items, root)

        with open(os.path.join(folder_for_files, file_name), "wb") as f:
            f.write(cls._prettify_xml(root))

    @classmethod
    def _prettify_xml(cls, elem):
        rough_string = ET.tostring(elem, 'utf-8')
        reparsed = minidom.parseString(rough_string)
        return reparsed.toprettyxml(indent="\t")

    @classmethod
    def _write_xml_node(cls, objects, root):
        if type(objects) is dict:
             data = ET.SubElement(root, 'data')
             for (k,v) in objects.items():
                 ET.SubElement(data, k).text = str(v)
        elif type(objects) is list:
            for l in objects:
                cls._write_xml_node(l, root)

    @classmethod
    def _write_xml_nodes(cls, items, root):
        for item in items:
            cls._write_xml_node(item.xml_data(root), root)


class XmlReader(object):
    """Utility class for reading data from XML files."""
    @classmethod
    def read_tabular_xml_file_into_list_of_dicts(cls, file_name_with_path):
        """
        :param file_name_with_path: absolute file path including .xml
        :type file_name_with_path: str
        :return: list of dicts

        Example usage of reading and writing to xml:

        .. code-block:: python

            from tropofy.file_io import read_write_xml

            # Example classes - could be a SQLA class or just a normal in memory Python class
            class Factory(read_write_xml.XmlInterfaceMixin):
                def __init__(name, max_employees):
                   self.name = name
                   self.max_employees = max_employees

            class Employee(read_write_xml.XmlInterfaceMixin):
                def __init__(name, factory_name):
                   self.name = name
                   self.factory_name = factory_name

            # Create some factory objects
            factories = [Factory('Factory 1', 100), Factory('Factory 1', 300)]

            # Write factories to XML
            folder_path = "/home/ubuntu/my_xml"
            read_write_xml.XmlWriter.write_xml_file(factories, "factories.xml", folder_path)

            # Run some code that assigns employees to factories

            #Read from XML.
            employees = [Employee(**row) for row in read_write_xml.XMLReader.read_tabular_xml_file_into_list_of_dicts(os.path.join(folder_path, 'Employees.xml'))]
        """
        try:
            tree = ET.parse(file_name_with_path)
        except Exception as e:
            raise e
        root = tree.getroot()
        rows = []
        for row in root:
            dict_ = {}
            for element in row:
                dict_[element.tag] = element.text
            if dict_:
                rows.append(dict_)
        return rows