'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from tropofy.widgets.widgets import Widget


class Chart(Widget):
    """Displays a chart on the GUI. This widget is an interface to `Google Charts <https://developers.google.com/chart/>`_. All Google Charts are supported.

    To create a chart, you must define the chart type, source data, and optionally provide additional formatting options for the chart.
    A charts source data is specified in table format. For each Chart, this table must take a specific format. For example, a pie chart
    expects a two-column table with a string column and a number column, where each row describes a slice, and the first column is the
    slice label and the second column is the slice value. A scatter chart, however, expects a table consisting of two numeric columns,
    where each row is a point, and the two columns are the X and Y values of the point.

    These formats are specified on the Google Charts documentation for each chart under Data Format (`Pie Chart example
    <https://developers.google.com/chart/interactive/docs/gallery/piechart#Data_Format>`_).

    Below is an example of configuring a Bar Chart:

    .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
       :pyobject: ExampleBarChart


    The following is a reference of the functions required to define a chart.

    """
    (AREACHART, BARCHART, BUBBLECHART, CANDLESTICKCHART, COLUMNCHART, COMBOCHART, HISTOGRAM, GAUGE, GEOCHART, LINECHART, ORGCHART, PIECHART, SCATTERCHART, STEPPEDAREACHART, TABLE, TIMELINE, TREEMAP) = (
        'AreaChart', 'BarChart', 'BubbleChart', 'CandleStickChart', 'ColumnChart', 'ComboChart', 'Histogram', 'Gauge', 'GeoChart', 'LineChart', 'OrgChart', 'PieChart', 'ScatterChart', 'SteppedAreaChart', 'Table', 'Timeline', 'TreeMap',
    )

    def __init__(self, widget_subscriptions=None):
        super(Chart, self).__init__(widget_subscriptions)

    def get_type(self):
        return "ChartWidget"

    def refresh_from_db(self, data_set, request):
        if data_set:
            return {
                'get_order_by_column': self.get_order_by_column(data_set),
                'get_column_ordering': self.get_column_ordering(data_set),
                'get_chart_options': self.get_chart_options(data_set),
                'get_table_data': self.get_table_data(data_set),
                'get_table_schema': self.get_table_schema(data_set),
                'get_chart_type': self.get_chart_type(data_set),
                'get_chart_title': self.get_chart_title(data_set),
                'get_chart_height': self.get_chart_height(data_set),
            }

    def get_chart_options(self, data_set):
        """(Optional) Each Chart has a range of configuration options. These are specified in the documentation for each chart under Configuration Options.

        :param data_set: Advanced use only. For programatically defining options, from the current ``data_set``.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: dictionary of options

        For example, a possible `set of options for a BarChart <https://google-developers.appspot.com/chart/interactive/docs/gallery/barchart#Configuration_Options>`_:

        .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
           :pyobject: ExampleBarChart.get_chart_options

        This will set the charts title to 'Company Performance', and draw a red title on the vertical axis labelled 'Year'.
        """
        return None

    def get_chart_type(self, data_set):
        """Specifies the type of chart to draw. All Google Charts are possible:

        * `AREACHART <https://developers.google.com/chart/interactive/docs/gallery/areachart>`_
        * `BARCHART <https://developers.google.com/chart/interactive/docs/gallery/barchart>`_
        * `BUBBLECHART <https://developers.google.com/chart/interactive/docs/gallery/bubblechart>`_
        * `CANDLESTICKCHART <https://developers.google.com/chart/interactive/docs/gallery/candlestickchart>`_
        * `COLUMNCHART <https://developers.google.com/chart/interactive/docs/gallery/columnchart>`_
        * `COMBOCHART <https://developers.google.com/chart/interactive/docs/gallery/combochart>`_
        * `GAUGE <https://developers.google.com/chart/interactive/docs/gallery/gauge>`_
        * `GEOCHART <https://developers.google.com/chart/interactive/docs/gallery/geochart>`_
        * `LINECHART <https://developers.google.com/chart/interactive/docs/gallery/linechart>`_
        * `ORGCHART <https://developers.google.com/chart/interactive/docs/gallery/orgchart>`_
        * `PIECHART <https://developers.google.com/chart/interactive/docs/gallery/piechart>`_
        * `SCATTERCHART <https://developers.google.com/chart/interactive/docs/gallery/scatterchart>`_
        * `STEPPEDAREACHART <https://developers.google.com/chart/interactive/docs/gallery/steppedareachart>`_
        * `TABLE <https://developers.google.com/chart/interactive/docs/gallery/table>`_
        * `TIMELINE <https://developers.google.com/chart/interactive/docs/gallery/timeline>`_
        * `TREEMAP <https://developers.google.com/chart/interactive/docs/gallery/treemap>`_

        :param data_set: Advanced use only. For programatically defining chart type, from the current ``data_set``.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: Chart.CHARTTYPE

        For example, for a bar chart:

        .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
           :pyobject: ExampleBarChart.get_chart_type

        """
        raise NotImplementedError

    def get_column_ordering(self, data_set):
        """(Optional - **though strongly recommended**, see note below) The order of the columns in the table used as a data source for the chart.

        :param data_set: Advanced use only. For programatically defining column ordering, from the current ``data_set``.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: list of strings, where the strings are all column IDs in the order in which you want the table created.

        For example, to define the order of columns for some table with columns IDs 'year', 'sales' and 'expenses':

        .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
           :pyobject: ExampleBarChart.get_column_ordering

        .. note:: You must list all column IDs in this parameter, if you use it. Furthermore, although optional,
            it is **strongly** recommended that you implement this function. Column order is very important for many charts,
            and there is no default order. The order in which you define the columns in :func:`tropofy.widgets.Chart.get_table_schema`
            is **not** a default ordering.
        """
        return None

    def get_order_by_column(self, data_set):
        """(Optional) The column to order the source table by. This ordering will often affect the way data is displayed on a chart.

        :param data_set: Advanced use only. For programatically defining order by column, from the current ``data_set``.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: str of column ID

        For example, to order a table by one of its columns 'year':

        .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
           :pyobject: ExampleBarChart.get_order_by_column
        """
        return None

    def get_table_data(self, data_set):
        """The rows of data of the charts data source table.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: List of dictionaries, where each dictionary represents a row of data. Each key must match a column ID specified in :func:`tropofy.widgets.Chart.get_table_schema`. 

        The simplest (though least useful) way to use this function is to return static data. The following is an example of returning static data. Each dictionary corresponds to
        a row in the table, with key value pairs where the keys correspond to the columns IDs (this table has the columns 'year', 'sales', and 'expenses'):

        .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
           :pyobject: ExampleBarChart.get_table_data

        The following example shows the data_set being queried to return a list of dictionaries of data. A user defined python class representing a store's
        financial 'Performance' is being queried, with data being summaried over each year for sales and expenses.

        .. literalinclude:: ../../../tropofy_example_apps/tutorials/tutorial_app_part_4.py
           :pyobject: PerformanceBarChart.get_table_data
        """
        raise NotImplementedError

    def get_table_schema(self, data_set):
        """The table schema describes all the columns in the table.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: dict. Key represents a column ID, and value a tuple describing the column. The tuple has the form ``(data_type [,label [,custom_properties]]])``.

        The following example uses a dictionary to define a static table schema of three columns with IDs: 'year', 'sales', and 'expenses'. The key: value
        pairs are in the format ID: (data_type, label).

        .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
           :pyobject: ExampleBarChart.get_table_schema

        Extensions to GoogleCharts:

        - Custom tooltips for Timeline charts. Add a column with role 'tooltip'. eg: "tooltip": ("string", "HoverOver", {'role': 'tooltip'})



        """
        raise NotImplementedError

    def get_chart_title(self, data_set):
        """Specify a title for the chart.

        .. note:: This title is independent of any title set in :func:`tropofy.widgets.Chart.get_chart_options`.
         It is recommended that this method only be used for charts where no title can be set through :func:`tropofy.widgets.Chart.get_chart_options`. For example, a TIMELINE chart.
         Furthermore, the height of this title is not included in the chart height specified in :func:`tropofy.widgets.Chart.get_chart_height`.

        :returns: Title for a chart
        :rtype: str
        """
        return None

    def get_chart_height(self, data_set):
        """Specify the height in px of the charts.

        Defaults to 600.

        :returns: Height of the chart in px
        :rtype: int
        """
        return 600


class FilteredChart(Chart):
    """A FilteredChart is a :class:`tropofy.widgets.Chart`, linked to a :class:`tropofy.widgets.Filter`.

    FilteredChart inherits from :class:`tropofy.widgets.Chart`, and is implemented identically except for the addition of a ``data_filter`` parameter and the helper function :func:`tropofy.widgets.FilteredChart.get_filter_values`.    

    :param data_filter: A filter widget that filters this chart.
    :type data_filter: :class:`tropofy.widgets.Filter`
    """
    def __init__(self, data_filter):
        self.filter = data_filter
        if self.filter:
            self.filter.add_event_subscription('refreshed', self.actions('show'))  # After filter initially loaded('refreshed'), show the chart.
            self.filter.add_event_subscription('formSubmitSuccess', self.actions('refresh'))

    def get_filter_values(self, data_set):
        """Helper method that returns the values of the filter. 

        Use this method within any methods of your :class:`tropofy.widgets.FilteredChart` with ``self.get_filter_values(data_set)``.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`

        :rtype: dict representation of the filtered values
        """
        try:
            return self.filter.get_values(data_set)
        except AttributeError:  # No filter.
            return {}

    def is_hidden(self):  # Makes chart hidden onload. Filter will trigger to show it when ready. Stops chart getting loaded incorrectly before filtered vars are set. If no filter, no need to hide chart.
        return self.filter is not None
