"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and governed by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

from sqlalchemy.types import Float, Text
from sqlalchemy.schema import Column, UniqueConstraint
from sqlalchemy.orm import exc
from tropofy.database.tropofy_orm import DataSetMixin
from tropofy.app import AppWithDataSets, Step, StepGroup
from tropofy.widgets import SimpleGrid, StaticImage, ExecuteFunction, Form

from PIL import Image
import io

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import odeint


class Scenario(DataSetMixin):
    name = Column(Text, nullable=False)
    initial_population = Column(Float, nullable=False)
    initial_zombie_population = Column(Float, nullable=False)
    initial_dead_population = Column(Float, nullable=False)
    birth_rate = Column(Float, nullable=False)
    natural_death_percent = Column(Float, nullable=False)
    transmission_percent = Column(Float, nullable=False)
    resurect_percent = Column(Float, nullable=False)
    destroy_percent = Column(Float, nullable=False)

    @classmethod
    def get_table_args(cls):
        return (UniqueConstraint('data_set_id', 'name'),)


class OutputPlot(StaticImage):
    def get_file_path(self, data_set):
        return data_set.get_image_path('output.png')


class SelectScenario(Form):
    def get_form_elements(self, data_set):
        scenarios = data_set.query(Scenario).all()
        scenario_names = [s.name for s in scenarios]

        # Example of using alternative text of select options.
        options = []
        for i, name in enumerate(scenario_names):
            options.append({
                'value': name,
                'text': 'Scenario %s: %s' % (i + 1, name)
            })

        default = data_set.get_var('scenario_name')
        if not default:
            options[0:0] = ['None']
        if options:
            return [Form.Element(
                name='scenario',
                input_type='select',
                default=default if default else options[0],
                label='Choose Scenario to Model',
                options=options,
                disabled=len(options) == 0
            )]

    def process_data(self, data, data_set):
        scenario_name = data.get('scenario')
        if scenario_name == 'None':
            return {
                'results': [{
                    'name': 'scenario',
                    'success': False,
                    'message': "'None' is not a valid scenario."
                }]
            }             
        data_set.set_var('scenario_name', scenario_name)

class VictoryLossImage(StaticImage):
    def get_file_path(self, data_set):
        zombie_victory = data_set.get_var('zombie_victory')
        if zombie_victory is not None:
            if zombie_victory:
                return 'https://s3-ap-southeast-2.amazonaws.com/tropofy.com/static/css/img/tropofy_example_app_icons/tombstone_zombie_victory.png'
            else:
                return 'https://s3-ap-southeast-2.amazonaws.com/tropofy.com/static/css/img/tropofy_example_app_icons/human_victory.png'


class ZombieOutbreakApp(AppWithDataSets):
    def get_name(self):
        return "Zombie Outbreak"

    def get_gui(self):
        return [
            StepGroup(
                name='Input',
                steps=[
                    Step(
                        name='Scenario Details',
                        widgets=[SimpleGrid(Scenario)],
                        help_text="Enter scenario details. All rates and percentages are per day."
                    ),
                    Step(
                        name='Model the Outbreak',
                        widgets=[SelectScenario(), CreatePlot()],
                    )
                ]
            ),
            StepGroup(
                name='Output',
                steps=[Step(
                    name='Output',
                    widgets=[
                        {'widget': OutputPlot(), 'cols': 6},
                        {'widget': VictoryLossImage(), 'cols': 6}
                    ],
                )]
            ),
        ]

    def get_examples(self):
        return {
            "Sample scenarios": load_sample_scenarios,
        }

    def get_icon_url(self):
        return 'https://s3-ap-southeast-2.amazonaws.com/tropofy.com/static/css/img/tropofy_example_app_icons/tombstone.png'

    def get_home_page_content(self):
        return {
            'content_app_name_header': '''
            <div>
            <span style="vertical-align: middle;">Zombie Outbreak</span>
            <img src="https://s3-ap-southeast-2.amazonaws.com/tropofy.com/static/css/img/tropofy_example_app_icons/tombstone.png" alt="main logo" style="width:15%">
            </div>''',

            'content_single_column_app_description': '''

            <p>Zombies zombies Zombies zombiesZombies zombies zombies zombiesZombies zombies</p>
            <p>This lighthearted app uses <a href="http://www.scipy.org/">Scipy</a>, <a href="http://www.numpy.org/">NumPy</a> and <a href="http://matplotlib.org/">Matplotlib</a>
            to model a Zombie Outbreak, as described by <a href='http://mysite.science.uottawa.ca/rsmith43/Zombies.pdf'>Munz et al. 2009</a>. You can model a range of different scenarios depending on the Zombie Outbreak that is affecting you!</p>

            ''',

            'content_row_4_col_1_content': '''
            This app was created using the <a href="http://www.tropofy.com" target="_blank">Tropofy platform</a>.
            '''
        }


class CreatePlot(ExecuteFunction):
    def get_button_text(self):
        return "Model the Apocalypse"

    def execute_function(self, data_set):
        scenario_name = data_set.get_var('scenario_name')
        try:
            scenario = data_set.query(Scenario).filter_by(name=scenario_name).one()
        except exc.NoResultFound:
            data_set.send_progress_message('No scenarios entered. Cannot model the apocalypse.')
            return

        plt.ion()

        P = scenario.birth_rate
        d = scenario.natural_death_percent
        B = scenario.transmission_percent
        G = scenario.resurect_percent
        A = scenario.destroy_percent

        def f(y, t):
            Si = y[0]
            Zi = y[1]
            Ri = y[2]

            # Munz et al. 2009, ODE equations.
            f0 = P - B*Si*Zi - d*Si
            f1 = B*Si*Zi + G*Ri - A*Si*Zi
            f2 = d*Si + A*Si*Zi - G*Ri
            return [f0, f1, f2]

        t = np.linspace(0, 10, 1000)  # time grid
        S0 = scenario.initial_population
        Z0 = scenario.initial_zombie_population
        R0 = scenario.initial_dead_population
        y0 = [S0, Z0, R0]

        # solve the DEs
        solution = odeint(f, y0, t)
        S = solution[:, 0]
        Z = solution[:, 1]
        #R = solution[:, 2]

        zombie_population = Z[-1]
        population = S[-1]
        data_set.set_var('zombie_victory', bool(zombie_population >= population))  # Need to convert to bool as otherwise evals to numpy.bool_ which isn't JSON serializable (i.e. data_set.set_var will throw error)

        plt.figure()
        plt.plot(t, S, label='Living')
        plt.plot(t, Z, label='Zombies')
        plt.xlabel('Days from outbreak')
        plt.ylabel('Population')
        plt.title('Zombie Apocalypse - Zombies vs. Humans')
        plt.legend(loc=0)

        buf = io.BytesIO()
        plt.savefig(buf, format="png", bbox_inches=0)
        buf.seek(0)
        img = Image.open(buf)
        data_set.save_image(name="output.png", image=img)
        data_set.send_progress_message("Apocalypse modelled for scenario '{scenario}'! Go to next step to view it. {note}".format(
            scenario=scenario_name,
            note="Warning: It doesn't look good for the humans..." if data_set.get_var('zombie_victory') else "You showed those Zombies!"
        ))


def load_sample_scenarios(data_set):
    data_set.add_all([
        Scenario(
            name='Doomed humans',
            initial_population=500,
            initial_zombie_population=0,
            initial_dead_population=0,
            birth_rate=0,
            natural_death_percent=0.0001,
            transmission_percent=0.0095,
            resurect_percent=0.0001,
            destroy_percent=0.0001,
        ),
        Scenario(
            name='Doomed zombies',
            initial_population=500,
            initial_zombie_population=200,
            initial_dead_population=0,
            birth_rate=0,
            natural_death_percent=0.0001,
            transmission_percent=0.0095,
            resurect_percent=0.0001,
            destroy_percent=0.015,
        )
    ])
