'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''
from tropofy.app import SynchronousTask

from tropofy.widgets import Widget
from sqlalchemy.orm import exc
from sqlalchemy.exc import InvalidRequestError
import traceback

from tropofy.app.data_set import CeleryTask
import collections
import transaction


class ExecuteFunction(Widget):
    """A button which executes a Python function when clicked.

    If your app solves a specific problem then this is where your custom code is executed.

    Below is an example of creating an ExecuteFunction widget:

    .. literalinclude:: ../../../tropofy_example_apps/core_dependencies/facility_location/facility_location.py
        :pyobject: ExecuteSolverFunction

    The following is a reference of the functions required to define an ExecuteFunction widget.
    """
    TYPE = "ExecuteFunctionWidget"

    def _get_type(self):
        return ExecuteFunction.TYPE

    def _refresh(self, request, data_set, **kwargs):
        return {
            'buttonText': self.get_button_text(),
            'executeTasksWithCelery': self.execute_tasks_with_celery(data_set),
            'maxSimultaneousTasksRunning': self.max_simultaneous_tasks_running(data_set),
            'maxTasksDisplayed': self.max_tasks_displayed(data_set),
            'progressMessageConsoleHeightPx': self.progress_message_console_height_px(data_set),
            'taskSpecs': [t.serialise() for t in data_set.get_tasks_qry(
                widget_unique_name=str(self.get_unique_name()),
                get_only_celery_tasks=self.execute_tasks_with_celery(data_set),
                limit=self.max_tasks_displayed(data_set),  # TODO: In future is could be possible to change this on the GUI. Would then have to send a param in the request
            )]
        }

    def _update(self, request, data, oper, data_set, **kwargs):
        if data_set is not None:
            task = None
            if oper == "init_synchronous_task":
                task = data_set.add(SynchronousTask(widget_unique_name=self.get_unique_name()))
            elif oper == "run_synchronous_task":
                self._run_synchronous_task(data_set, task_id=int(data['taskId']))
                return  # Nothing returned by synchronous task - would be a long running process which we don't want to keep track of. Poll task instead.
            elif oper == "start_celery_tasks":
                tasks = self._start_celery_tasks(data_set)
                return [t.serialise() for t in tasks]  # Celery can start multiple tasks at once.
            else:  # oper == "request_progress":
                task = data_set.get_task(data['taskId'])
                if oper == "terminate_task":
                    task.terminate()

            return task.serialise()
        return {}

    # Note because the developers solver function is executed as part of a SPECIFIC ajax request it is already in a thread of it's own!
    def _run_synchronous_task(self, data_set, task_id):
        """Runs a synchronous task. Not using Celery"""
        synchronousTask = None
        data_set_id = data_set.id  # Cache in case data_set becomes detached from session
        try:
            synchronousTask = data_set.query(SynchronousTask).filter_by(id=task_id).one()
        except exc.NoResultFound:
            print("Synchronous task not initialised correctly. Please try again and if problems persists, you may need to refresh the page.")  # TODO: Make better than print statement?
            return
        try:
            data_set._running_task = synchronousTask
            self.execute_function(data_set)
            synchronousTask.track_task_finished(execution_success=True)
            # data_set._running_task = None
        except Exception as e:
            messages = []
            db_session = data_set.db_session
            try:
                messages = synchronousTask.messages
            except InvalidRequestError:
                pass
            print(traceback.format_exc())
            transaction.abort()
            synchronousTask = db_session().query(SynchronousTask).filter_by(id=task_id, data_set_id=data_set_id).one()
            synchronousTask._messages = messages
            synchronousTask.track_task_finished(execution_success=False, exception=e)

    def _start_celery_tasks(self, data_set):
        """Can start one or multiple tasks within execute function."""
        async_results = self.execute_function(data_set)
        if not isinstance(async_results, collections.Iterable):
            async_results = [async_results]

        tasks = []
        for result in list(async_results):
            if type(result).__name__ != 'AsyncResult':  # Weird comparision. type(result) == AsyncResult doesn't work for some reason so resorting to using str.
                 raise Exception("For async task execution, execute_function must return instance of or list of AsyncResult.")
            tasks.append(CeleryTask(
                task_id=result.id,
                widget_unique_name=self.get_unique_name()
            ))
        data_set.add_all(tasks)
        return tasks

    def get_button_text(self):  # TODO: Add data_set on here so can make dynamic (won't be back compatible though and people will have to change code in apps)
        """The text on the button in the GUI.

        :rtype: a string

        .. literalinclude:: ../../../tropofy_example_apps/core_dependencies/facility_location/facility_location.py
            :pyobject: ExecuteSolverFunction.get_button_text
            :language: python
        """
        return 'Execute Function'

    def execute_function(self, data_set):
        """Any Python code to be executed.

        :param data_set: The current ``data_set`` object through which the database is accessed.
        :type data_set: :class:`tropofy.app.AppDataSet`

        :rtype: None

        Use :func:`tropofy.app.AppDataSet.send_progress_message` to post messages to the GUI.

        Example: The following example demonstrates implementing code to be executed. This code assumes a SQLA ORM class ``Location`` exists.

        .. code-block:: python

          def execute_function(self, data_set):
              data_set.send_progress_message("Deleting locations...")
              data_set.query(Location).delete()
              data_set.send_progress_message("Locations deleted.")
        """
        raise NotImplementedError

    def execute_tasks_with_celery(self, data_set):
        """Execute tasks asynchronously and receive messages during task execution. Additional configuration is required beyond returning ``True`` here - see 'Working With Celery' Tropofy docs.

        :param data_set: The current ``data_set`` object through which the database is accessed.
        :type data_set: :class:`tropofy.app.AppDataSet`

        :rtype: bool
        """
        return False

    def max_simultaneous_tasks_running(self, data_set):
        """It is possible to run multiple tasks simultaneously. Defaults to 1.

        :param data_set: The current ``data_set`` object through which the database is accessed.
        :type data_set: :class:`tropofy.app.AppDataSet`

        :rtype: int
        """
        return 1

    def max_tasks_displayed(self, data_set):
        """The n most recently started tasks will have their details displayed. Defaults to 1.

        :param data_set: The current ``data_set`` object through which the database is accessed.
        :type data_set: :class:`tropofy.app.AppDataSet`

        :rtype: int
        """
        return 1

    def progress_message_console_height_px(self, data_set):
        """Height in px for the task progress message console.

        Default is 300. Example below sets height to 600.

        .. code-block:: python

          def progress_message_console_height_px(self, data_set):
              return 600

        :param data_set: The current ``data_set`` object through which the database is accessed.
        :type data_set: :class:`tropofy.app.AppDataSet`

        :rtype: int
        """
        return 300