'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from tropofy.widgets.widgets import Widget
from pyramid.response import Response
from pyramid import threadlocal


class CustomExport(Widget):
    """Displays a download button in the GUI to download custom content.

    The following is a reference of the functions required to define a Export.
    """
    def _get_type(self):
        return "CustomExportWidget"

    def _refresh(self, request, data_set, **kwargs):
        if data_set:
            return {'success': True, 'data': self.get_custom_export_data(data_set)}

    def _update(self, request, data, oper, data_set, **kwargs):
        response = Response(self.get_custom_export_data(data_set))
        response.headers['Content-Disposition'] = (("attachment; filename=%s"%(self.get_custom_export_file_name_with_extension(data_set))))
        return response
    
    def _serialise(self, **kwargs):
        '''Join serialisation dict with Widget to get all params needed on client'''
        return dict(Widget._serialise(self, **kwargs).items() + {
            'buttonText': self.get_export_button_text()
        }.items())

    def get_custom_export_file_name_with_extension(self, data_set):
        """The name of the file to be downloaded.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :returns: file name with extension
        :rtype: str
        """
        raise NotImplementedError

    def get_custom_export_data(self, data_set):
        """The data to be downloadable.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :returns: valid data string
        """
        raise NotImplementedError
    
    def get_export_button_text(self):
        """The text on the button in the GUI.

        :rtype: str
        """
        return 'Export'
    
    
class ExportDataSet(CustomExport):
    """ExportDataSet. This widget is special. It is not used like the other widgets. You do not need to create a class
    which implements function to use an instance of this widget. You can simply use it as it is.
    """
    
    def get_custom_export_file_name_with_extension(self, data_set):
        return "ExportedDataSet.xlsx"

    def get_custom_export_data(self, data_set):
        return threadlocal.get_current_request().app.get_data_set_as_excel_wb_string_repr(data_set)
    
    def get_export_button_text(self):
        return 'Export Data Set'
    
