'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from tropofy.widgets.widgets import Widget
from pyramid.response import Response


class KMLMap(Widget):
    """Displays a map and a download button in the GUI. The map displays the contents of a KML file and the download button enables it to be downloaded.

    To create a map you simply implement a function which returns the KML you want displayed. The map will only display static KML features.
    You can generate the KML content however you like but we recommend using the `SimpleKml <https://code.google.com/p/simplekml/>`_ package
    which is included with Tropofy. The SimpleKML documentation can be found `here <http://simplekml.readthedocs.org/en/latest/>`_

    Below is an example of creating a KMLMap:

    .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
        :pyobject: MyKMLMap

    The following is a reference of the functions required to define a KMLMap.

    """
    def _get_type(self):
        return "KMLMapWidget"

    def _refresh(self, request, data_set, **kwargs):
        if data_set:
            return {'kml': self.get_kml(data_set)}

    def _update(self, request, data, oper, data_set, **kwargs):
        response = Response(self.get_kml(data_set))
        response.headers['Content-Disposition'] = (("attachment; filename=KML_shown_on_map.kml"))
        return response

    def get_kml(self, data_set):
        """The KML to be downloadable and displayed on the map

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: valid KML as a string

        The following example shows the data_set being queried to return a list of Locations and Lines.
        Each location has a longitude and latitude member. Lines have a start and end location. KML mixins
        representing each Location and Line are created, added to a KML folder and given a specific style

        .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
            :pyobject: MyLocsLinesKMLMap.get_kml
        """
        raise NotImplementedError
