'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

import requests
import json
from tropofy.database.tropofy_orm import ExtendedJsonEncoder
import pkg_resources
import pyramid.httpexceptions as exc
from pyramid.response import Response
from pyramid.security import authenticated_userid, remember, forget
from pyramid.view import view_config, forbidden_view_config, notfound_view_config


def route_url_to_gui_server(request):
    return request.registry.settings.get('custom.gui_server_url') + request.path


def proxy_request(request, data=None, url=None):
    data = data if data else {}
    return post_to_gui_server(
        request=request,
        url=url if url else route_url_to_gui_server(request),
        data=data
    )


def post_to_gui_server(request, url, data):
    app = request.app
    key = app.get_published_license_key() if app and app.get_published_license_key() else request.registry.settings.get('custom.license_key')
    core_data = {
        'license_key': key,
        'authenticated_user': authenticated_userid(request),
        'client_ip': request.client_addr,
        'tropofy_version': pkg_resources.get_distribution("tropofy").version,
    }

    if app:
        core_data.update({
            'app_name': app.name,
            'app_icon_url': app.get_icon_url(),
            'page_title': app.get_page_title(),
            'force_mfa_for_all_users': app.force_mfa_for_all_users(),
        })

    core_data.update(data)  # So it's possible to override core_data attributes in data.
    
    response = requests.post(url, data=json.dumps(core_data, cls=ExtendedJsonEncoder), headers={'Content-type': 'application/json', 'Accept': 'text/plain'})
    check_response_for_success(response)
    return response


def log_activity(request, description):
    url = request.registry.settings.get('custom.gui_server_url') + '/' + request.app.url_name + '/log_activity'
    proxy_request(
        request=request,
        data={'description': description, 'from_compute_node': True},
        url=url
    )


@view_config(route_name='run_app', permission='run_app')
def run_app(request):
    app = request.app
    result = proxy_request(request, data={
        'step_groups': [step_group.serialise(app=app, authenticated_user=authenticated_userid(request)) for step_group in app.get_ordered_step_groups()],
        'app_init_config': app._get_app_init_config(authenticated_user=authenticated_userid(request)),
    })
    return Response(result.content)


def get_url_location(request, route_name, app_url_name):
    if request.registry.settings.get('custom.force_redirects_to_https'):
        # Need to do this because current AWS setup routes through LB to HTTP so request.schema always HTTP
        return request.route_url(route_name, app_url_name=app_url_name, _scheme='HTTPS', _port=443)
    return request.route_url(route_name, app_url_name=app_url_name, _scheme=request.scheme, _port=request.host_port)


@view_config(route_name='redirect_to_run_app')
def redirect_to_run_app(request):
    if request.only_app:
        return exc.HTTPFound(location=get_url_location(request, route_name='run_app', app_url_name=request.only_app.url_name))
    else:
        raise exc.HTTPNotFound()


@view_config(route_name='update_widget', renderer='json', permission='run_app')
def update_widget(request):
    response = request.widget._update(request=request, data=dict(request.POST), oper=request.matchdict['oper'], data_set=request.data_set)
    if request.matchdict['oper'] == 'needs_text_response':
        request.response.content_type = 'text/plain'
    return response


@view_config(route_name='refresh_widget', renderer='json')  # , xhr=True, renderer='json', permission='run_app')
def refresh_widget(request):
    widget_parameters = request.widget._refresh(request=request, data_set=request.data_set)
    if request.widget._get_type() is "KMLMapWidget":  # too much to send up and back
        response = Response(widget_parameters['kml'])
        response.content_type = 'text/xml'  # Must explicitly set for ie
        return response
    elif request.widget._get_type() is "CustomExportWidget":  # too much to send up and back
        response = Response(widget_parameters['data'])
        response.content_type = 'text/plain'  # Must explicitly set for ie
        return response
    else:
        return widget_parameters


@view_config(route_name='export_data_set', permission='run_app')
def export_data_set(request):
    if request.data_set_id:
        log_activity(request, ("Export data set: '{name}'".format(name=request.data_set.name)))
    else:
        log_activity(request, 'Export Excel Template')
    return request.app.export_data_set(request.data_set)


### Data Set Management ###
@view_config(route_name='create_new_data_set', xhr=True, renderer='json', permission='run_app')
def create_new_data_set(request):
    app = request.app
    if app is not None:
        data_set = app.create_new_data_set_for_user(
            authenticated_userid(request),
            request.POST['data_set_name'],
        )
        app.populate_data_set_from_template(data_set, request.POST['data_set_template'])
        return {
            'success': True,
            'dataSetId': data_set.id
        }
    return {'success': False}


@view_config(route_name='start_app_with_data_set', xhr=True, renderer='json', permission='run_app')
def start_app_with_data_set(request):
    data_set = request.data_set
    data_set.update_time_accessed()
    return {
        'currentStepIndex': data_set.current_step_index,
        'latestEnabledStepIndex': data_set.latest_enabled_step_index,
        'dataSetId': data_set.id,
        'dataSetName': data_set.name,
    }


@view_config(route_name='set_data_set_current_step', xhr=True, renderer='json', permission='run_app')
def set_data_set_current_step(request):
    current_step_index = int(request.matchdict['current_step_index'])
    if request.data_set:  # some apps have no data set
        if current_step_index > 0:  # Don't store if current step is data_set_selection step.
            request.data_set.current_step_index = current_step_index
            if request.data_set.latest_enabled_step_index < current_step_index:
                request.data_set.latest_enabled_step_index = current_step_index
    return {'success': True}


@view_config(route_name='login', renderer='json', request_method='POST')
def login(request):
    data = {
        'login': request.POST.get('login'),
        'password': request.POST.get('password'),
        'totp_token': request.POST.get('totp_token'),
    }

    if proxy_request(request, data).status_code == 200:
        headers = remember(request, request.POST['login'])
        request.response.headerlist.extend(headers)
        return {'success': True}


@forbidden_view_config(renderer='string')
@view_config(route_name='logout', renderer='string')
def logout(request):
    headers = forget(request)
    data = request.app.get_home_page_configuration()
    data.update({
        'login_screen_default_is_create_account': request.app.is_login_screen_default_create_account(),
    })
    result = proxy_request(
        request,
        data=data,
        url=request.registry.settings.get('custom.gui_server_url') + '/' + request.app.url_name + '/homepage',
    )
    return Response(result.content, headers=headers)


@view_config(context=exc.HTTPUnauthorized)
@view_config(route_name='manage_account', renderer='string')
def manage_account(request):
    url = request.registry.settings.get('custom.gui_server_url') + '/' + request.app.url_name + '/manage_account'
    result = proxy_request(request, url=url)
    return Response(result.content)


@view_config(context=exc.HTTPNotAcceptable)
@view_config(route_name='manage_mfa', renderer='string')
def manage_mfa(request):
    url = request.registry.settings.get('custom.gui_server_url') + '/' + request.app.url_name + '/manage_mfa'
    result = proxy_request(request, url=url)
    return Response(result.content)


# @forbidden_view_config(renderer='string')
# def forbidden(request):
#     return exc.HTTPFound(location=get_url_location(request, route_name='logout', app_url_name=request.matchdict.get('app_url_name')))


@notfound_view_config()
def notfound(exc, request):
    """Catches all 'raise HTTPNotFound'.

    Todo: Plain text reponse for now. Should render to a template. Template should be on GUI server though so need to proxy.
    """
    response_str = "Tropofy 404: We're sorry, but the page you are looking for could not be found."
    if request.matchdict and request.matchdict.get('app_url_name'):
        response_str += " You should check the spelling of the app you are trying to access: " + \
            "'" + request.matchdict['app_url_name'] + "'"
    response = Response(response_str)
    response.status = 404
    return response


def check_response_for_success(response):
    # (1) HTTPForbidden thrown on GUI server
    if response.status_code == 403:
        raise exc.HTTPForbidden()

    # (2) HTTPUnauthorized thrown on GUI server - using Temp password and have to change it.
    if response.status_code == 401:
        raise exc.HTTPUnauthorized()

    # (3) HTTPNotFound thrown on GUI server - e.g. if app_name is not valid
    if response.status_code == 404:
        raise exc.HTTPNotFound()

    # (4) HTTPNotAcceptable thrown on GUI server - users must be using MFA and user isn't.
    if response.status_code == 406:
        raise exc.HTTPNotAcceptable()

    # (5) NonCompatibleComputeNodeException thrown on GUI server.
    if response.status_code == 409:
        pass  # Let response get rendered.

    if response.status_code == 500:
        raise Exception(response.text)

    # (6) Tropofy caught failure
    success = True
    message = ""
    try:
        response = json.loads(response.text)
        success = response.get('success')
        message = response.get('message')
    except:
        pass
    if not success and success is not None:
        raise Exception(message)


@view_config(route_name='favicon')
def favicon(response):
    return Response()


@view_config(route_name='healthcheck')
def healthcheck(response):
    return Response()
