"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

from tropofy.database.db_manager import DbManager
from tropofy.widgets import Form
from tropofy.widgets import grid_widget
from sqlalchemy.sql.expression import desc, or_, cast
from sqlalchemy import String
from sqlalchemy.orm import _mapper_registry
from sqlalchemy.schema import CheckConstraint
from tropofy.database import DBSession


class SimpleGrid(grid_widget.GridWidget):
    """Basic Grid. This widget is special. It is not used like the other widgets. You do not need to create a class
    which implements function to use an instance of this widget.

    :param source_class: A user defined SQLAlchemy Python class
    :param editable: (Optional - default True). Enables grid to be edited.
    :type editable: bool
    :param widget_subscriptions: (Optional) Widget to subscribe to a triggered event of
    :type widget_subscriptions: list of tuples of (tropofy.widget, str of trigger_event_name, str of subscribed_action)

    .. note::
        The source_class provided to a SimpleGrid must have the following properties:
            - The database column names must be the same as the member variable names, i.e. you cannot use code like ``some_name = Column('some_other_name', String(50))``),
            - The initialisers parameter names must be the same as the member variable names. Note that SQLAlchemy provides a default initialiser with keyword arguments equal to the names of the column members defined for your class.

    The following class satisfies the above requirements

    .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
        :pyobject: ConformingClass
    """
    def __init__(self, source_class, editable=True, title=None, embedded_form_class=None, desc_sort_cols=None, widget_subscriptions=None):
        self.source_class = source_class
        self.editable = editable
        super(SimpleGrid, self).__init__(
            title=title,
            embedded_form_class=embedded_form_class if embedded_form_class else SimpleGridCreateUpdateForm,
            desc_sort_cols=desc_sort_cols,
            widget_subscriptions=widget_subscriptions
        )

    def _get_type(self):
        return 'SimpleGrid'

    def add_new_row(self, data, data_set_id):
        edited_obj = DbManager.construct_object_and_add_to_db(source_class=self.source_class, data=data, data_set_id=data_set_id)
        return edited_obj.as_json_data_row(self.get_grids_column_names())

    def edit_row(self, data):
        edited_obj = DbManager.update_object_in_db(source_class=self.source_class, data=data)
        return edited_obj.as_json_data_row(self.get_grids_column_names())

    def delete_row(self, obj_id):
        DbManager.delete_object_from_db(source_class=self.source_class, obj_id=obj_id)

    def get_table_hierachy_from_inheritance_chain(self):
        full_class_hierachy = list(self.source_class.__bases__) + [self.source_class]
        return [class_type for class_type in full_class_hierachy if hasattr(class_type, "__table__")]

    def get_grids_column_names(self):
        return [col.name for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns]

    def get_column_name_to_form_input_types(self):
        return {col.name: Form._get_input_type_from_python_type(col.type.python_type) for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns}

    def get_filtered_rows(self, data_set, display_start, display_length, global_search_field, ordered_col_name_sort_direction_tuples):
        try:
            qry = self.get_qry_of_non_filtered_refresh_objects(data_set)
        except AttributeError:
            # TODO: Log that there was an attribute error on the refreshish
            return [], 0, 0
        total_records_before_filtering = qry.count()

        #Filter from search
        if global_search_field:
            search_conditions = []
            do_not_search_on_cols = self.get_col_names_to_not_search_on(data_set)
            for col_name in DbManager.sqla_cls_defined_column_names(self.source_class):
                if col_name not in do_not_search_on_cols:
                    search_conditions.append(
                        cast(getattr(self.source_class, col_name), String).ilike('%' + global_search_field + '%')  # .ilike is case insensitive .like
                    )
            qry = qry.filter(or_(*search_conditions))
                    
        total_records_after_filtering = qry.count()
        
        #Order cols
        for col_name, direction in ordered_col_name_sort_direction_tuples:
            if direction == grid_widget.GridWidget.ASCENDING:
                qry = qry.order_by(getattr(self.source_class, col_name))
            elif direction == grid_widget.GridWidget.DESCENDING:
                qry = qry.order_by(desc(getattr(self.source_class, col_name)))

        #Reduce to num rows on grid page
        qry = qry.offset(display_start).limit(display_length)
        rows = [obj.as_json_data_row(self.get_grids_column_names()) for obj in qry]
        return rows, total_records_before_filtering, total_records_after_filtering

    def get_qry_of_non_filtered_refresh_objects(self, data_set):
        """
        :returns: SQLA qry object, that if executed, would return all mixins in the grid - unfiltered and unsorted.
        
        Assumes data_set exists.
        """
        if data_set and hasattr(self.source_class, 'data_set_id'):
            return data_set.query(self.source_class)
        raise AttributeError('SimpleGrid for source_class `%s` has no `data_set_id` column. SimpleGrid expects this. If not using data sets, extend SimpleGrid and overide `get_qry_of_non_filtered_refresh_objects`' % self.source_class.__name__)

    def get_col_names_to_not_search_on(self, data_set):
        return self.get_hidden_column_names()

    def grid_is_editable(self, **kwargs):
        return self.editable


class SimpleGridCreateUpdateForm(grid_widget.CreateUpdateForm):
    def get_select_options_for_column(self, data_set, col_name, hidden_col_names):
        """For a column, get the select options for the form element. For a SimpleGrid - base off of foreign keys on tables."""
        options = None

        col_name_to_foreign_key_col_objects = {name: col for (name, col) in self._get_query_column_objects().items() if col.foreign_keys and col.name not in hidden_col_names}
        col_name_to_check_constraint_allowable_values = self.get_check_constraints_where_column_must_be_in_list_of_values(data_set)

        if col_name in col_name_to_foreign_key_col_objects.keys():
            column = col_name_to_foreign_key_col_objects[col_name]
            foreign_key_col = list(column.foreign_keys)[0].column
            mapper = _get_mapper_for_table(foreign_key_col.table)

            if data_set:
                options_qry = data_set.query(getattr(mapper.class_, foreign_key_col.name))
            else:
                options_qry = DBSession().query(getattr(mapper.class_, foreign_key_col.name))
            options = [Form.Element.Option(v[0]) for v in options_qry]

            if column.nullable:
                options = [Form.Element.Option(value=None)] + options

        elif col_name in col_name_to_check_constraint_allowable_values.keys() and col_name not in hidden_col_names:
            options = [Form.Element.Option(v) for v in col_name_to_check_constraint_allowable_values[col_name]]

        return options

    def get_check_constraints_where_column_must_be_in_list_of_values(self, data_set):
        dict_cons = {}
        col_names = self._get_query_column_objects().keys()
        for con in self.parent_grid_widget.source_class.__table__.constraints:
            if type(con) is CheckConstraint and hasattr(con, 'sqltext') and hasattr(con.sqltext, 'text'):
                con_text = con.sqltext.text
                col_name = con_text.split()[0] if con_text.split() and con_text.split()[0] in col_names else None
                if col_name and con_text.find("in") != -1:
                    values_text = con_text[con_text.find("(")+1:con_text.find(")")]
                    dict_cons[col_name] = [v.replace("'", "") for v in values_text.split(",")]
        return dict_cons

    def _get_query_column_objects(self):
        return {col.name: col for class_type in self.parent_grid_widget.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns}


def _get_mapper_for_table(table):
    for mapper in list(_mapper_registry):
        if table in mapper._all_tables:
            return mapper
    return None