from tropofy.widgets import Widget


class FilteredWidgetMixin(Widget):
    """A FilteredChart is a :class:`tropofy.widgets.Chart`, linked to a :class:`tropofy.widgets.Filter`.

    FilteredChart inherits from :class:`tropofy.widgets.Chart`, and is implemented identically except for the addition of a ``data_filter`` parameter and the helper function :func:`tropofy.widgets.FilteredChart.get_filter_values`.

    :param data_filter: A filter widget that filters this chart.
    :type data_filter: :class:`tropofy.widgets.Filter`
    """
    def __init__(self, data_filter, **kwargs):
        self.filter = data_filter
        if self.filter:
            self.filter.add_event_subscription('refreshed', self.actions('show'))  # After filter initially loaded('refreshed'), show the chart.
            self.filter.add_event_subscription('formSubmitSuccess', self.actions('refresh'))
        super(FilteredWidgetMixin, self).__init__(**kwargs)

    def get_filter_values(self, data_set):
        """Helper method that returns the values of the filter.

        Use this method within any methods of your :class:`tropofy.widgets.FilteredChart` with ``self.get_filter_values(data_set)``.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`

        :rtype: dict representation of the filtered values
        """
        try:
            return self.filter.get_values(data_set)
        except AttributeError:  # No filter.
            return {}

    def is_hidden(self, **kwargs):  # Makes chart hidden onload. Filter will trigger to show it when ready. Stops chart getting loaded incorrectly before filtered vars are set. If no filter, no need to hide chart.
        return self.filter is not None
