'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''
from collections import namedtuple, defaultdict


class WidgetConfigException(Exception):
    """An exception raised if an obvious error has occured in configuring a widget"""
    pass


class Widget(object):
    """A widget is a HTML interface to the database via a developer defined query or framework specified query

    :param widget_subscriptions: (Optional) Widget to subscribe to a triggered event of
    :type widget_subscriptions: list of tuples of (tropofy.widget, str of trigger_event_name, str of subscribed_action)
    """
    widget_indexer = 0
    widget_dict = {}

    Action = namedtuple('Action', 'widget name')
    # all_actions = ['refresh', 'hide', 'show']

    def __init__(self, widget_subscriptions=None):
        if widget_subscriptions:
            for widget_subscription in widget_subscriptions:
                widget, trigger_event_name, subscribed_action = widget_subscription

                widget.add_event_subscription(
                    trigger_event_name=trigger_event_name,
                    subscribed_actions=[
                        self.actions(subscribed_action),
                    ]
                )

    def actions(self, name):
        """Return an event object to pass to add_event_subscription"""
        return Widget.Action(self, name)

    def add_event_subscription(self, trigger_event_name, subscribed_actions):
        """Add subscription(s) to a trigger event on this widget.

        :param trigger_event_name: 
        :param subscribed_actions: 
        :type subscribed_actions: a Widget.Action or list of Widget.Action's.
        """
        if not hasattr(self, 'event_subscriptions'):  # Create dict: event_name: [list of subscriptions]
            self.event_subscriptions = defaultdict(list)
        if type(subscribed_actions) is not list:
            subscribed_actions = [subscribed_actions]
        # Add checking that trigger_event_name is a valid event. Perhaps with an enum pattern?
        self.event_subscriptions[trigger_event_name] += subscribed_actions

    def _allocate_id(self):
        self.id = Widget.widget_indexer
        Widget.widget_indexer += 1
        assert self.id not in Widget.widget_dict
        Widget.widget_dict[self.id] = self
        self.num_cols_in_gui = 12

    def _get_type(self):
        '''Unique identifier for the widget'''
        pass

    def _update(self, request, data, oper, data_set, **kwargs):
        """Update the widget - not a full refresh."""
        pass

    def _refresh(self, request, data_set, **kwargs):
        """Refresh all data for widget."""
        pass

    def _add_embedded_widget(self, name, widget):
        
        if not hasattr(self, 'embedded_widgets'):
            self.embedded_widgets = {}

        widget._allocate_id()
        self.embedded_widgets.update({name: widget})

    __mapper_args__ = {  # TODO: Remove these? No longer using SQLA here....
        'polymorphic_identity': 'widget',
        'polymorphic_on': type
    }

    def _serialise(self, **kwargs):
        '''Dict representation for easy conversion to JSON.'''
        event_subscriptions = self.event_subscriptions if hasattr(self, 'event_subscriptions') else {}
        serialised_event_subscriptions = {}
        for event_name, subscribed_actions in event_subscriptions.iteritems():
            serialised_event_subscriptions.update({event_name: []})
            for action in subscribed_actions:
                serialised_event_subscriptions[event_name].append({
                    'widget_id': action.widget.id,
                    'name': action.name
                })

        return {
            'id': self.id,
            'type': self._get_type(),
            'numColsInGui': self.num_cols_in_gui,
            'embeddedWidgetSpecs': {key: value._serialise(**kwargs) for key, value in self.embedded_widgets.iteritems()} if hasattr(self, 'embedded_widgets') else {},
            'eventSubscriptions': serialised_event_subscriptions,
            'isHidden': self.is_hidden(**kwargs),
            'enablePrint': self.enable_print(**kwargs),
        }

    def is_hidden(self, **kwargs):
        """Hides a widget. Can be shown through events. Default: False.

        Warning: this is evaluated once on app launch. After this, visibility must be managed by events.
        :returns: bool
        """
        return False

    def get_unique_name(self):
        """Return a name for the widget that is unique in this app.

         This name can be used to uniquely reference the widget in database tables. This is used by :class:`tropofy.widgets.ExecuteFunction` to store task history relative to a widget.

        .. warning: It is up to the developer to ensure names are unique across widgets in an app. If this method is not supplied, the default is an auto generated ID which will change
         when widgets are added or moved in the workflow. This can be confusing in development, and limits the ability to quickly change production apps without confusion.
         """
        return str(self.id)

    def enable_print(self, **kwargs):
        """Shows a button on the widget to print it."""
        # TODO: Make widget accept **kwargs. Then pass a 'printable' option.
        return False