"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""
import json
from datetime import datetime, date, time, timedelta


class JsonStrFormats():
    TIME = '%H:%M:%S'  # TODO: Should make these ISO compliant. Would make hooking into 3rd party libs easier (eg Google Charts).
    DATE = '%Y-%m-%d'
    DATETIME = '%Y-%m-%d %H:%M:%S'


class ExtendedJsonEncoder(json.JSONEncoder):
    """Custom JSON encoding. Matching decoding...

    By overriding 'default', we can provide custom JSON encoding (date, times, etc.). However, Pyramid also does this,
    (e.g. to enable ORMAncestor.__json__) and will actually override our ExtendedJsonEncoder.default function. It does
    this, by providing a value for 'default' to json.JSONEncoder.__init__().

    Below we override json.JSONEncoder.__init__() to intercept the default param to ensure our default is called. The
    value of default is stored in self.second_default so that it is called after ExtendedJsonEncoder.default.
    """
    def __init__(self, skipkeys=False, ensure_ascii=True, check_circular=True, allow_nan=True, sort_keys=False, indent=None, separators=None, encoding='utf-8', default=None):

        self.second_default = default

        super(ExtendedJsonEncoder, self).__init__(skipkeys=skipkeys, ensure_ascii=ensure_ascii,
            check_circular=check_circular, allow_nan=allow_nan, sort_keys=sort_keys,
            indent=indent, separators=separators, encoding=encoding, default=None)

    def default(self, obj):
        if isinstance(obj, time):
            return obj.strftime(JsonStrFormats.TIME)
        elif isinstance(obj, datetime):  # Must come before 'date' as isinstance(datetime_obj, date) == True
            return obj.strftime(JsonStrFormats.DATETIME)
        elif isinstance(obj, date):
            return obj.strftime(JsonStrFormats.DATE)
        elif isinstance(obj, timedelta):
            return str(obj)

        else:
            if self.second_default:
                return self.second_default(obj)
            else:
                return super(ExtendedJsonEncoder, self).default(obj)

# Custom json_renderer for Pyramid
from pyramid.renderers import JSON
json_renderer = JSON(cls=ExtendedJsonEncoder)
