"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""
import json
from datetime import datetime, date, time
from sqlalchemy.types import Integer, Text, Unicode, Float, DateTime, Date, Time, Boolean
import custom_json


class ValidationException(Exception):
    """"""
    def __init__(self, name=None, value=None, message=None):
        self.message = "Validation error"
        if name is not None:
            self.message += ": %s" % name
            if value is not None:
                self.message += " = '%s'" % str(value)
        self.message += '.'
        if message is not None:
            self.message += ' %s' % message

    def __str__(self):
        return self.message


class Validator():

    @classmethod
    def validate(cls, value, type, name=None):
        '''Validates that 'value' is of type 'type'. Converts if able. 'name' helps give meaningful error msg's.'''        
        validator = cls._get_validator(type)
        if value is not None and validator:
            return validator(value, name)
        else:
            return value

    @classmethod
    def _get_validator(cls, validating_type):
        if type(validating_type) is type:
            validating_type = validating_type.__name__

        return type_to_validator_mapping.get(validating_type)

    @classmethod
    def _validate_int(cls, value, name):
        if isinstance(value, basestring):
            value = value.replace(',', '')
        if isinstance(value, float):
            value = int(round(value))
        if not isinstance(value, int):
            try:
                value = int(value)
            except:
                try:  # For values such as '1.0'. int('1.0') raises exception. float('1.0') doesn't.
                    value = int(round(float(value)))
                except:
                    pass
        if not isinstance(value, int):
            raise ValidationException(name=name, value=value, message='Expected integer')
        return value

    @classmethod
    def _validate_float(cls, value, name):
        if isinstance(value, basestring):
            value = value.replace(',', '')
        if not isinstance(value, float):
            try:
                value = float(value)
            except:
                pass
        if not isinstance(value, float):
            raise ValidationException(name=name, value=value, message='Expected decimal')
        return value

    @classmethod
    def _validate_string(cls, value, name):
        """
        Validates is str or unicode. SQLAlchemy 'Text' can store either. Should be using unicode wherever possible as this allow for non ascii chars.
        Converting to str(value) will try and convert to ascii (very limited char set). In Python 3, 'str' is actually 'unicode' and old 'str' functionality is 'bytes'
        """
        try:
            if type(value) is not unicode:
                value = str(value)
        except:
            raise ValidationException(name=name, value=value, message='Expected string')
        return value

    @classmethod
    def _validate_boolean(cls, value, name):
        """Boolean is subclass of int in Python. False == 0, True == 1"""
        try:  # Will convert "false" and "true" to bool. Help with processing bools from js input.
            value = json.loads(value)
        except:
            pass
        try:
            value = bool(value)
        except:
            raise ValidationException(name=name, value=value, message='Expected boolean')
        return value

    @classmethod
    def _validate_time(cls, value, name):
        if isinstance(value, basestring):
            value = value.strip()
            try:
                value = datetime.strptime(value, custom_json.JsonStrFormats.TIME).time()
            except:
                try:
                    value = datetime.strptime(value, '%H:%M').time()
                except:
                    pass
        if not isinstance(value, time):        
            raise ValidationException(name=name, value=value, message='Expected time')
        return value

    @classmethod
    def _validate_date(cls, value, name):
        if isinstance(value, basestring):
            value = value.strip()
            try:
                value = datetime.strptime(value, custom_json.JsonStrFormats.DATE).date()
            except Exception as e:
                try:
                    value = datetime.strptime(value, '%d/%m/%Y').date()
                except:
                    try:
                        value = datetime.strptime(value, '%Y-%m-%d').date()
                    except:
                        pass

        if isinstance(value, datetime):
            value = value.date()
        if not isinstance(value, date):
            raise ValidationException(name=name, value=value, message='Expected date')
        return value

    @classmethod
    def _validate_datetime(cls, value, name):
        if isinstance(value, basestring):
            value = value.strip()
            try:
                value = datetime.strptime(value, custom_json.JsonStrFormats.DATETIME)
            except ValueError:
                try:
                    value = datetime.strptime(value, "%H:%M %d/%m/%Y")
                except:
                    pass
            except:
                pass
        if not isinstance(value, datetime):
            raise ValidationException(name=name, value=value, message='Expected datetime')
        return value


type_to_validator_mapping = {
    Integer: Validator._validate_int,
    Float: Validator._validate_float,
    Text: Validator._validate_string,
    Unicode: Validator._validate_string,
    Date: Validator._validate_date,
    Time: Validator._validate_time,
    DateTime: Validator._validate_datetime,
    Boolean: Validator._validate_boolean,
    'int': Validator._validate_int,
    'float': Validator._validate_float,
    'str': Validator._validate_string,
    'date': Validator._validate_date,
    'time': Validator._validate_time,
    'datetime': Validator._validate_datetime,
    'bool': Validator._validate_boolean,
}
