'''
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
'''

from sqlalchemy.exc import IntegrityError, StatementError
from tropofy.widgets import Widget, WidgetConfigException, Form
from tropofy.database.tropofy_orm import ValidationException


class GridWidget(Widget):
    """
    Display data in a grid.

    :param title: A title for the grid.
    :type title: str
    :param desc_sort_cols: cols to sort descending
    :type desc_sort_cols: list of col names
    """
    ASCENDING = 'asc'
    DESCENDING = 'desc'
    CUSTOM_MAPPERS = {
        'format_date': 'formatDate',
        'format_time': 'formatTime',
        'format_datetime': 'formatDateTime',
    }

    def __init__(self, title=None, embedded_form_class=None, desc_sort_cols=None, widget_subscriptions=None):
        embedded_form_class = embedded_form_class if embedded_form_class else CreateUpdateForm
        create_update_form = embedded_form_class(parent_grid_widget=self)
        self._add_embedded_widget(name='createUpdateForm', widget=create_update_form)
        create_update_form.add_event_subscription('formSubmitSuccess', [self.actions('standingRefresh'), self.actions('hideCreateUpdateForm')])

        self.title = title
        self.desc_sort_cols = desc_sort_cols if desc_sort_cols else []

        super(GridWidget, self).__init__(widget_subscriptions=widget_subscriptions)

    def _get_type(self):
        raise NotImplementedError()

    def _refresh(self, request, data_set, **kwargs):
        col_names = self.get_grids_column_names()
        ordered_col_name_sort_direction_tuples = []
        for i in xrange(len(col_names)):
            try:
                ordered_col_name_sort_direction_tuples.append((
                    col_names[int(request.POST['order[%s][column]' % i])],
                    request.POST['order[%s][dir]' % i],
                ))
            except KeyError:
                break

        rows, total_records_before_filtering, total_records_after_filtering = self.get_filtered_rows(
            data_set=data_set,
            display_start=int(request.POST['start']),
            display_length=int(request.POST['length']),
            global_search_field=request.POST['search[value]'],
            ordered_col_name_sort_direction_tuples=ordered_col_name_sort_direction_tuples,
        )
        if rows == [] or rows:  # [] means no data in table = OK to proceed. None or something else falsy means an error and shouldn't proceed.
            return {
                'draw': int(request.POST['draw']),  # Suggested to cast to int to protect against cross-site-scripting attacks,
                'recordsTotal': total_records_before_filtering,
                'recordsFiltered': total_records_after_filtering,
                'data': rows,
            }

    def _update(self, request, data, oper, data_set, **kwargs):
        if oper == 'del':
            self.delete_row(obj_id=data['id'])
        return {'success': True}

    def _serialise(self, **kwargs):
        """Join serialisation dict with Widget to get all params needed on client"""
        return dict(Widget._serialise(self, **kwargs).items() + {
            'columns': self.get_grids_column_names(),
            'hiddenCols': self.get_hidden_column_names(),
            'descSortedCols': self.get_decs_sorted_cols(),  # Cols default is to sort ascending on init. If in this list they will start sorted descending.
            'nonEditabledCols': self.get_non_editable_cols(),
            'gridIsEditable': self.grid_is_editable(**kwargs),
            'title': self.get_title(),
            'colIndexToCustomFormatMaps': self._get_custom_formatting_mappers(),
            'datetimeDisplayFormats': self._get_datetime_display_formats(),
        }.items())

    def grid_is_editable(self, **kwargs):
        raise NotImplementedError

    def get_grids_column_names(self, **kwargs):
        raise NotImplementedError

    def get_non_editable_cols(self, **kwargs):
        """Will disable input boxes for these cols on edit in the GUI."""
        return []

    def get_decs_sorted_cols(self, **kwargs):  # TODO: 'desc' is typo - should be 'desc'
        return self.desc_sort_cols

    def get_hidden_column_names(self, **kwargs):
        return ['id', 'data_set_id']

    def get_filtered_rows(self, data_set, display_start, display_length, global_search_field, ordered_col_name_sort_direction_tuples, **kwargs):
        """
        Rows returned is a list of lists. Each list represents a row of data in the grid. The elements of the row must be in the same order as the column names.

        :returns: (filtered list of lists of row data to display in grid for this page/search, total records in data set before filter, total records in data set after filter)
        :rtype: tuple - (<list of lists of data>, <int>, <int>)
        """
        raise NotImplementedError        

    def edit_row(self, data, **kwargs):
        raise NotImplementedError

    def add_new_row(self, data, data_set_id=None, **kwargs):
        raise NotImplementedError

    def delete_row(self, obj_id, **kwargs):
        raise NotImplementedError

    def get_title(self, **kwargs):
        """Display a title for the grid. Set once for the grid - can not be dynamically changed.
        :rtype: str
        """
        return self.title

    def get_column_name_to_form_input_types(self, **kwargs):
        """Input types to use in the form for cols. Values must be in Form.Element.VALID_INPUT_TYPES.

        If not specified for column, will default to Form.Element.TEXT"""
        return {}

    def get_column_name_to_form_default(self, **kwargs):
        """Default values to use in the form corresponding to each column. Keys must correspond to column names.

        :return: dict of {column_name: default_value}
        """
        return {}

    def _get_datetime_display_formats(self):
        formats = {
            'date': "DD/MM/YYYY",
            'datetime': "HH:mm DD/MM/YYYY",
            'time': "HH:mm"
        }
        custom_formats = self.get_datetime_display_formats()
        if set(custom_formats.keys()) > Form.Element.VALID_DATETIME_INPUT_TYPES:
            raise WidgetConfigException("Error in function Form.get_datetime_display_formats. Must return dict with keys in: [Form.Element.DATE, Form.Element.TIME, Form.Element.DATETIME]")
        formats.update(custom_formats)
        return formats

    def get_datetime_display_formats(self, **kwargs):
        """Specify custom display formats to display datetimes, dates, and times in a grid and its edit form.

        Use format strings specified by `MomentJS <http://momentjs.com/docs/#/parsing/string-format/>`_. Note that these are slightly different to string-formats used in Python.

        Allowable keys are: 'date', 'datetime', and 'time'

        :returns: dict
        """
        return {}

    def _get_custom_formatting_mappers(self):
        col_names_to_form_input_types = self.get_column_name_to_form_input_types()
        col_names = self.get_grids_column_names()

        # Setup default mappers
        form_input_type_to_mapper = {
            'date': GridWidget.CUSTOM_MAPPERS['format_date'],
            'time': GridWidget.CUSTOM_MAPPERS['format_time'],
            'datetime': GridWidget.CUSTOM_MAPPERS['format_datetime'],
        }

        mappers = {name: form_input_type_to_mapper[col_names_to_form_input_types[name]] for name in col_names if col_names_to_form_input_types[name] in form_input_type_to_mapper.keys()}

        # Override with custom mappers
        custom_mappers = self.get_custom_formatting_mappers()
        if set(custom_mappers.keys()) > set(self.get_grids_column_names()):
            raise WidgetConfigException("Error in function Grid.get_custom_formatting_mappers. Must return dict with keys in columns names: %s" % col_names)

        mappers.update(custom_mappers)

        return {col_names.index(name): mapper for name, mapper in mappers.iteritems()}

    def get_custom_formatting_mappers(self, **kwargs):  # TODO: Write a custom mapper to put ',' in numbers and use as example.1
        """Customise the format of a particular column in the grid and edit form.

        Access mappers with ``GridWidget.CUSTOM_MAPPERS['mapper_name']

        Available Mappers:
         - 'format_date': Format date in display format specified in :func:`Grid.get_datetime_display_formats`
         - 'format_time': Format time in display format specified in :func:`Grid.get_datetime_display_formats`
         - 'format_datetime': Format datetime in display format specified in :func:`Grid.get_datetime_display_formats`

        :returns: dict of {col_name: mapper}
        """
        return {}


class CreateUpdateForm(Form):
    def __init__(self, parent_grid_widget):
        self.parent_grid_widget = parent_grid_widget
        super(CreateUpdateForm, self).__init__()

    def get_select_options_for_column(self, data_set, col_name, hidden_col_names):
        """For a column, get the select options for the form element."""
        return None

    def get_form_elements(self, data_set, **kwargs):
        if self.parent_grid_widget.grid_is_editable():  # TODO: Fix Jono's Hack to stop super large foreign key select drop downs loading in non editable grid. Nicer to remove whole form. Nicer again to maybe use typeahead with dynamic options... or something for FK's.
            form_elements = []
            non_editable_col_names = self.parent_grid_widget.get_non_editable_cols()
            hidden_col_names = self.parent_grid_widget.get_hidden_column_names()
            for name in self.parent_grid_widget.get_grids_column_names():
                input_type = self.parent_grid_widget.get_column_name_to_form_input_types().get(name, Form.Element.TEXT)
                default = self.parent_grid_widget.get_column_name_to_form_default().get(name, None)

                options = self.get_select_options_for_column(data_set, name, hidden_col_names)
                if options:
                    input_type = Form.Element.SELECT
                form_elements.append(Form.Element(
                    name=name,
                    input_type=input_type,
                    hidden=name in hidden_col_names,
                    disabled=name in non_editable_col_names,
                    options=options,
                    default=default,
                ))
            return form_elements
        return []

    def process_data(self, data, data_set, **kwargs):
        message = ''
        success = True
        for k, v in data.iteritems():  # Convert empty string entries to None. Needed to validate against is nullable on strings. (empty strings used on non nullable elsewhere in framework. Long term fix is to change this!)
            if v == '':
                data[k] = None
        if data['id']:  # Edit existing object
            try:
                self.parent_grid_widget.edit_row(data)
            except (ValidationException, IntegrityError) as e:
                message = e.message
                success = False
            except StatementError as e:  # StatementError thrown when ValidationException on encrypted col.
                message = e.orig.message  # Get original ValidationException message
                success = False
        else:  # Add new object
            try:
                self.parent_grid_widget.add_new_row(data, data_set.id if data_set else None)
            except (ValidationException, IntegrityError) as e:
                message = e.message
                success = False
        self.post_process_data(success=success, data=data, data_set=data_set)
        return {'success': success, 'message': message}

    def is_autosubmit(self, data_set, **kwargs):
        return False

    def get_datetime_display_formats(self, **kwargs):
        return self.parent_grid_widget._get_datetime_display_formats()

    def post_process_data(self, success, data, data_set, **kwargs):
        """method to custom process data after objects edited, added, or deleted

        :param success: whether or not modification was successful
        :type success: bool
        :param data: data passed to :func:`tropofy.widgets.grid_widget.grid_widget.CreateUpdateForm.process_data`
        :param data_set:
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param kwargs:
        :return:
        """
        pass
