"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

import collections
from sqlalchemy import event
from sqlalchemy.schema import Column, ForeignKey, ForeignKeyConstraint
from sqlalchemy.types import Integer
from sqlalchemy.ext.declarative import declared_attr

# Use * imports to make back compatible and so only need to import from tropofy.database.tropofy_orm
from custom_json import *
from base import *
from validation import *
from encryption import *


class DynamicSchemaName(object):
    use_app_schemas = None
    schema_name = None
    COMPUTE_NODE_SCHEMA_NAME = 'tropofy_compute'
    schema_name_has_been_used = False  # Track for assertion. Need to ensure when use_app_schemas is set, that get_schema_qualified_table_name_if_using_schemas hasn't been called.

    @classmethod  # Would be good to be a @property, but apparently can't do @property setters on classmethods.
    def set_schema_name(cls, value):
        """
        It is the responsibility of the developer to ensure two apps don't use the same schema unless this is desired.
        :type value: str
        """
        if not cls.use_app_schemas:
            cls.schema_name = None
        else:
            cls.schema_name = value

    @classmethod  # Would be good to be a @property, but apparently can't do @property setters on classmethods.
    def set_use_app_schemas(cls, value):
        """
        :type value: bool
        """
        assert not cls.schema_name_has_been_used, 'DynamicSchemaName.use_app_schemas accessed before assigned. DynamicSchemaName.set_use_app_schemas must be called before any table creation to ensure schemas are used correctly. This is why some imports (Especially AppDataSet) are called within main.'
        cls.use_app_schemas = value

    @classmethod
    def get_schema_qualified_table_name_if_using_schemas(cls, table_name):
        cls.schema_name_has_been_used = True
        return '%s%s' % (
            DynamicSchemaName.COMPUTE_NODE_SCHEMA_NAME + '.' if DynamicSchemaName.use_app_schemas else '',
            table_name
        )


class TableConstructionException(Exception):
    pass


class TropofyDbMetaData(object):
    metadata = base.ORMBase.metadata

    TableNameAndSchema = collections.namedtuple('TableNameAndSchema', ['name', 'schema'])

    @classmethod
    def get_dependency_sorted_tables(cls, table_name_and_schemas):
        """
        Returns list of tables that match a given table name and schema.

        :type table_name_and_schemas: list of TropofyDbMetaData.TableNameAndSchema

        :returns: list of sqlalchemy.sql.schema.Table
        """
        return [table for table in cls.metadata.sorted_tables if (table.name, table.schema) in table_name_and_schemas]


class DynamicSchemaMixin(base.ORMBase):
    """Used as mixin with DataSetMixin and separately for apps not using data sets.."""
    __abstract__ = True

    @declared_attr
    def __table_args__(cls):
        __table_args__ = cls.get_table_args()

        if DynamicSchemaName.use_app_schemas:
            __table_args__ += ({'schema': DynamicSchemaName.schema_name},)

            # Add schema to ForeignKeyConstraint's
            for arg in __table_args__:
                if type(arg) is ForeignKeyConstraint:
                    for col, fk in arg._elements.items():
                        arg._elements[col] = ForeignKey(
                            DynamicSchemaName.schema_name + '.' + fk._colspec,
                            _constraint=arg,
                            name=arg.name,
                            onupdate=arg.onupdate,
                            ondelete=arg.ondelete,
                            use_alter=arg.use_alter,
                            link_to_name=arg.link_to_name,
                            match=arg.match
                        )
        return __table_args__

    @classmethod
    def get_table_args(cls):
        return ()


class DataSetMixin(DynamicSchemaMixin):
    """Used as mixin for all developer defined models."""
    __abstract__ = True

    @declared_attr
    def data_set_id(cls):
        return Column(Integer, ForeignKey(DynamicSchemaName.get_schema_qualified_table_name_if_using_schemas('appdataset.id'), ondelete='CASCADE'))


# this event is called whenever an attribute
# on a class is instrumented
@event.listens_for(base.ORMAncestor, 'attribute_instrument')
def configure_listener(class_, key, inst):
    if not hasattr(inst.property, 'columns'):
        return
    # this event is called whenever a "set"
    # occurs on that instrumented attribute

    @event.listens_for(inst, "set", retval=True)
    def set_(instance, value, oldvalue, initiator):
        col = inst.property.columns[0]
        if value == '' and col.type.python_type in [int, float]:
            value = None
        if value is None and col.default:  # If a default exists, set that to value if the value is None.
            value = col.default.arg
        if value is None and not col.nullable:
            py_type = col.type.python_type
            expected_type_str = 'decimal' if py_type is float else py_type.__name__
            raise validation.ValidationException(name=col.name, value=value, message='Expected %s' % expected_type_str)
        return validation.Validator.validate(value, col.type.__class__, col.name)


def enable_sqlite_foreign_keys():
    """ Enable foreign_keys in sqlite. This is called when no db is explicitly defined, meaning sqlite is used.
    In memory sqlite db for decrypting things benefits from NOT having FK's enforced (especially on grids).
    """
    from sqlalchemy.engine import Engine
    from sqlalchemy import event
    import sqlite3


    @event.listens_for(Engine, "connect")
    def set_sqlite_pragma(dbapi_connection, connection_record):
        if type(dbapi_connection) is sqlite3.Connection:
            cursor = dbapi_connection.cursor()
            cursor.execute("PRAGMA foreign_keys=ON")
            cursor.close()
