"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""
import inspect
from abc import ABCMeta
from sqlalchemy.schema import Column
from sqlalchemy.types import Integer
from sqlalchemy.ext.declarative import declarative_base, declared_attr, DeclarativeMeta
import json
import custom_json


class ORMAncestor(object):
    id = Column(Integer, primary_key=True)

    @declared_attr
    def __tablename__(cls):
        return cls.__name__.lower()

    def _base_blacklist(self):
        '''Blacklist list of which properties not to include in JSON'''
        return ['_sa_instance_state']

    def _json_blacklist(self):
        '''Blacklist to overide'''
        return []

    def __json__(self, request, fields_to_expand=[]):  # http://stackoverflow.com/questions/5022066/how-to-serialize-sqlalchemy-result-to-json & https://gist.github.com/pjenvey/3808830
        """Converts object to json ready dict. Need to add encoders for non Json ready types (such as Date)."""
        blacklist = set(self._base_blacklist())
        blacklist.update(self._json_blacklist())

        dict_repr = {}
        for k, v in self.__dict__.items():
            if k not in blacklist:
                try:
                    # 2) Need to use only functions by name otherwise they return as "<function get_kml_for_downloading at 0x0000000005364048>" where the mem address changes
                    if isinstance(v, DeclarativeMeta):
                        v = str(v)
                    elif inspect.isroutine(v):
                        v = v.func_name
                    json.dumps(v, cls=custom_json.ExtendedJsonEncoder)  # this will fail on non-encodable values, like other classes
                    dict_repr[k] = v
                except TypeError:
                    dict_repr[k] = None

        # a json-encodable dict
        return dict_repr

    def as_json_data_row(self, ordered_field_names):
        '''Returns json safe row of field vals in order of ordered_field_names.'''
        fields = self.__json__(None)
        row = []
        for field in ordered_field_names:
            value = fields.get(field, None)  # Will be None if can't match col or value in db is None.
            row.append(value if value is not None else '')
        return row


class AllowABCsToBeInheritedFromInMultipleInheritanceCases(ABCMeta, DeclarativeMeta):
    pass


ORMBase = declarative_base(cls=ORMAncestor, metaclass=AllowABCsToBeInheritedFromInMultipleInheritanceCases)