"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and governed by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

from tropofy.widgets.widgets import Widget
import os
from tropofy.file_io import read_write_xl


class FileUpload(Widget):
    """A button through which files can be uploaded to an app.

    A progress bar is displayed, as well as results for all files uploaded. Files are processed on a compute node
    with :func:`tropofy.widgets.FileUpload.process_file`. Multiple file selected is not supported by some older browsers (ie8).
    """

    def _get_type(self):
        return "FileUploadWidget"

    def _get_file_size(self, file_in):
        file_in.seek(0, 2)  # Seek to the end of the file
        size = file_in.tell()  # Get the position of EOF
        file_in.seek(0)  # Reset the file position to the beginning
        return size

    def _validate(self, data_set, result):
        max_file_size = self.get_max_file_size(data_set)
        if result['size'] > max_file_size:
            result['message'] = 'File exceeds max file size of {max}MB. File size is {actual}MB'.format(
                max=max_file_size/1000000,
                actual=result['size']/1000000,
            )
        elif result['type'] not in self.get_allowed_file_types(data_set):
            result['message'] = 'File type not allowed.'
        else:
            return True
        result['success'] = False
        return False

    def _update(self, request, data, oper, data_set, **kwargs):  # Validate and process files. If multiple files uploaded at once, they are processed one by one.
        results = []
        for _, in_memory_file in request.POST.items():
            if isinstance(in_memory_file, unicode):  # SND: What does this do?
                continue
            result = {
                'name': os.path.basename(in_memory_file.filename),
                'type': in_memory_file.type,
                'size': self._get_file_size(in_memory_file.file),
            }

            if self._validate(data_set, result):
                try:
                    self.process_file(data_set, in_memory_file, result)
                    result['success'] = True
                except Exception as e:
                    result['success'] = False
                    if hasattr(e, 'history_as_list') and e.history_as_list:
                        result['message'] = e.history_as_list
                    elif hasattr(e, 'message') and e.message:
                        result['message'] = e.message
                    elif hasattr(e, 'msg') and e.msg:
                        result['message'] = e.msg
            results.append(result)
        return results

    def process_file(self, data_set, in_memory_file, result, **kwargs):
        """Process an in memory file.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :param in_memory_file: An uploaded file in memory to be processed.
        :type in_memory_file: FieldStorage
        :param result: A dictionary to store result information. Report a message by setting result['message'].
           A single string can be added or a list of strings giving more detailed information.
        :type result: dict
        :rtype: None
        """
        raise NotImplementedError

    def get_allowed_file_types(self, data_set, **kwargs):
        """List of allowable file types to be uploaded.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: list of allowable file types. Each type is a `MIME type <http://en.wikipedia.org/wiki/Internet_media_type#Type_application>`_.
        """
        raise NotImplementedError

    def get_max_file_size(self, data_set, **kwargs):
        """Max uploadable file size (bytes). Default 10mb.

        :param data_set: The DataSet object on which queries can be made to access the apps data.
        :type data_set: :class:`tropofy.app.AppDataSet`
        :rtype: int (number of bytes)
        """
        return 10000000  # 10 MB

    excel_file_types = [
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'application/vnd.ms-excel.sheet.macroEnabled.12',
        'application/vnd.ms-excel',
        'application/x-zip-compressed',
        'application/octet-stream',
    ]


class ClassDataFileUpload(FileUpload):

    def __init__(self, data_classes):
        if isinstance(data_classes, list):
            self.data_classes = data_classes
        else:
            self.data_classes = [data_classes]

    def get_allowed_file_types(self, data_set, **kwargs):
        return FileUpload.excel_file_types

    def process_file(self, data_set, in_memory_file, result, **kwargs):
        if not data_set:
            raise Exception("There is no data set. You will have to create one before you can upload any data.")
        for data_class in self.data_classes:
            data_set.query(data_class).delete()
        result['message'] = read_write_xl.ExcelReader.load_data_from_excel_file_in_memory(
            data_set,
            in_memory_file,
            ordered_class_ws_mappings=self.get_class_ws_mappings(),
        )

    def get_class_ws_mappings(self):
        return [data_class.get_class_ws_mapping() for data_class in self.data_classes]